 //  ####  ######    ####  #######   ####  ------------------------- //
//  ##  ##  ##  ##  ##  ##  ##   #  ##  ##  CPCEC, plain text Amstrad //
// ##       ##  ## ##       ## #   ##       CPC emulator written in C //
// ##       #####  ##       ####   ##       as a postgraduate project //
// ##       ##     ##       ## #   ##       by Cesar Nicolas-Gonzalez //
//  ##  ##  ##      ##  ##  ##   #  ##  ##  since 2018-12-01 till now //
 //  ####  ####      ####  #######   ####  ------------------------- //

// The ZILOG Z80A that runs inside the Amstrad CPC is very similar to
// other Z80 chips used in 8-bit and 16-bit computers and consoles from
// the 1980s and 1990s. Its timings are slightly different but making
// them fit other machines only requires new timing macros and tables.

// This module also includes two optional debuggers. One is graphical;
// the other one is based on text commands similar to MS-DOS DEBUG.EXE.

// BEGINNING OF Z80 EMULATION ======================================= //

WORD z80_wz; // internal register WZ/MEMPTR
BYTE z80_iff0=0,z80_halted=0; // internal HALT flag

BYTE z80_flags_inc[256],z80_flags_dec[256]; // INC,DEC
BYTE z80_flags_sgn[256],z80_flags_add[512],z80_flags_sub[512]; // ADD,ADC,SUB,SBC...
BYTE z80_flags_and[256],z80_flags_xor[256],z80_flags_bit[256]; // AND,XOR,OR,BIT...

WORD z80_break_stack=0xFFFF;
BYTE z80_debug_peekpoke=0;
#ifdef DEBUG
#define z80_debug_reset()
#define z80_debug_close()
#else
BYTE z80_breakpoints[1<<16];
WORD z80_debug_volatile=0;
WORD z80_debug_pnl0_w=0; // code byte / shadow PC
WORD z80_debug_pnl3_w=0; // stack word / shadow SP
BYTE z80_debug_logtmp[1<<9];
WORD z80_debug_logpos;
FILE *z80_debug_logfile=NULL;
void z80_debug_reset(void)
{
	z80_debug_pnl0_w=z80_pc.w;
	z80_debug_pnl3_w=z80_sp.w;
	debug_buffer[0]=128; // 128: draw debug screen
}
void z80_debug_close(void)
{
	if (z80_debug_logfile)
	{
		if (z80_debug_logpos)
			fwrite(z80_debug_logtmp,1,z80_debug_logpos,z80_debug_logfile);
		fclose(z80_debug_logfile);
		z80_debug_logfile=NULL;
	}
}
#endif

void z80_setup(void) // setup the Z80
{
	#ifdef DEBUG
	#else
		MEMZERO(z80_breakpoints);
	#endif
	// flag bit reference:
	// - 7, 0x80: S, Sign
	// - 6, 0x40: Z, Zero
	// - 5, 0x20: 5, undocumented
	// - 4, 0x10: H, Half carry
	// - 3, 0x08: 3, undocumented
	// - 2, 0x04: V, Parity/oVerflow
	// - 1, 0x02: N, add/substract
	// - 0, 0x01: C, Carry
	// 9-bit expression RESULT^OP1^OP2 : bit 8 = full Carry, bit 7 = Sign carry, bit 4 = Half carry
	int i;
	for (i=0;i<256;++i) // build flags that depend on the 8-bit result: S, Z and sometimes H and V too
	{
		int p=i&1;
		if (i&2)
			++p;
		if (i&4)
			++p;
		if (i&8)
			++p;
		if (i&16)
			++p;
		if (i&32)
			++p;
		if (i&64)
			++p;
		if (i&128)
			++p;
		p=(p&1)?0x00:0x04; // -----P--
		int j=i?(i&0x80):0x40; // SZ------
		int x=j+(i&0x28); // SZ5-3----
		z80_flags_sgn[i]=x; // SZ5-3---
		z80_flags_inc[i]=x+(i==128?0x04:0)+((i&15)==00?0x10:0); // SZ5H3V0-
		z80_flags_dec[i]=x+(i==127?0x04:0)+((i&15)==15?0x12:2); // SZ5H3V1-
		z80_flags_and[i]=0x10+(z80_flags_xor[i]=(x+p)); // SZ513P--,SZ503P--
		z80_flags_bit[i]=0x10+j+p; // SZ-1-P--
		j=(i&0x10)+((i&128)>>5); // ---H-V--
		z80_flags_sub[i]=2+(z80_flags_add[i]=j); // ---H-V!0
		z80_flags_sub[256+i]=2+(z80_flags_add[256+i]=j^5); // ---H-V!1
	}
}

void z80_reset(void) // reset the Z80
{
	z80_irq=z80_halted=z80_pc.w=z80_sp.w=z80_iff.w=z80_iff0=z80_imd=0;
	z80_ix.w=z80_iy.w=0xFFFF;
}

// Z80 disassembler ------------------------------------------------- //

enum { // list of opcodes and parameters used by the dictionnaries (cfr. infra)
	Z80_DASM_NULL	,	Z80_DASM_SKIP	,	Z80_DASM_BACK	,	Z80_DASM_ADC	,
	Z80_DASM_ADD	,	Z80_DASM_AND	,	Z80_DASM_BIT	,	Z80_DASM_CALL	,
	Z80_DASM_CCF	,	Z80_DASM_CP	,	Z80_DASM_CPD	,	Z80_DASM_CPDR	,
	Z80_DASM_CPI	,	Z80_DASM_CPIR	,	Z80_DASM_CPL	,	Z80_DASM_DAA	,
	Z80_DASM_DEC	,	Z80_DASM_DI	,	Z80_DASM_DJNZ	,	Z80_DASM_EI	,
	Z80_DASM_EX	,	Z80_DASM_EXX	,	Z80_DASM_HALT	,	Z80_DASM_IM	,
	Z80_DASM_IN	,	Z80_DASM_INC	,	Z80_DASM_IND	,	Z80_DASM_INDR	,
	Z80_DASM_INI	,	Z80_DASM_INIR	,	Z80_DASM_JP	,	Z80_DASM_JR	,
	Z80_DASM_LD	,	Z80_DASM_LDD	,	Z80_DASM_LDDR	,	Z80_DASM_LDI	,
	Z80_DASM_LDIR	,	Z80_DASM_NEG	,	Z80_DASM_NOP	,	Z80_DASM_OR	,
	Z80_DASM_OTDR	,	Z80_DASM_OTIR	,	Z80_DASM_OUT	,	Z80_DASM_OUTD	,
	Z80_DASM_OUTI	,	Z80_DASM_POP	,	Z80_DASM_PUSH	,	Z80_DASM_RES	,
	Z80_DASM_RET	,	Z80_DASM_RETI	,	Z80_DASM_RETN	,	Z80_DASM_RL	,
	Z80_DASM_RLA	,	Z80_DASM_RLC	,	Z80_DASM_RLCA	,	Z80_DASM_RLD	,
	Z80_DASM_RR	,	Z80_DASM_RRA	,	Z80_DASM_RRC	,	Z80_DASM_RRCA	,
	Z80_DASM_RRD	,	Z80_DASM_RST	,	Z80_DASM_SBC	,	Z80_DASM_SCF	,
	Z80_DASM_SET	,	Z80_DASM_SLA	,	Z80_DASM_SLL	,	Z80_DASM_SRA	,
	Z80_DASM_SRL	,	Z80_DASM_SUB	,	Z80_DASM_XOR	,	Z80_DASM_BYTE	,
	Z80_DASM_WORD	,	Z80_DASM_P_B_	,	Z80_DASM_P_W_	,	Z80_DASM_P_BC	,
	Z80_DASM_P_C	,	Z80_DASM_P_DE	,	Z80_DASM_P_HL	,	Z80_DASM_P_IX	,
	Z80_DASM_Q_IX	,	Z80_DASM_P_SP	,	Z80_DASM_A	,	Z80_DASM_AF	,
	Z80_DASM_AF2	,	Z80_DASM_B	,	Z80_DASM_BC	,	Z80_DASM_C	,
	Z80_DASM_D	,	Z80_DASM_DE	,	Z80_DASM_E	,	Z80_DASM_H	,
	Z80_DASM_HL	,	Z80_DASM_I	,	Z80_DASM_IX	,	Z80_DASM_L	,
	Z80_DASM_NC	,	Z80_DASM_NS	,	Z80_DASM_NV	,	Z80_DASM_NZ	,
	Z80_DASM_R	,	Z80_DASM_S	,	Z80_DASM_SP	,	Z80_DASM_V	,
	Z80_DASM_XH	,	Z80_DASM_XL	,	Z80_DASM_Z	,	Z80_DASM_0	,
	Z80_DASM_1	,	Z80_DASM_2	,	Z80_DASM_DIV8	,	Z80_DASM_R_PC	,
};
const char z80_dasm_lex[][5]= // SAME ORDER AS ABOVE ENUM!!
{
	""	,	"*NOP"	,	"*NOP"	,	"ADC"	,
	"ADD"	,	"AND"	,	"BIT"	,	"CALL"	,
	"CCF"	,	"CP"	,	"CPD"	,	"CPDR"	,
	"CPI"	,	"CPIR"	,	"CPL"	,	"DAA"	,
	"DEC"	,	"DI"	,	"DJNZ"	,	"EI"	,
	"EX"	,	"EXX"	,	"HALT"	,	"IM"	,
	"IN"	,	"INC"	,	"IND"	,	"INDR"	,
	"INI"	,	"INIR"	,	"JP"	,	"JR"	,
	"LD"	,	"LDD"	,	"LDDR"	,	"LDI"	,
	"LDIR"	,	"NEG"	,	"NOP"	,	"OR"	,
	"OTDR"	,	"OTIR"	,	"OUT"	,	"OUTD"	,
	"OUTI"	,	"POP"	,	"PUSH"	,	"RES"	,
	"RET"	,	"RETI"	,	"RETN"	,	"RL"	,
	"RLA"	,	"RLC"	,	"RLCA"	,	"RLD"	,
	"RR"	,	"RRA"	,	"RRC"	,	"RRCA"	,
	"RRD"	,	"RST"	,	"SBC"	,	"SCF"	,
	"SET"	,	"SLA"	,	"SLL"	,	"SRA"	,
	"SRL"	,	"SUB"	,	"XOR"	,	"\001"	,
	"\002"	,	"\003"	,	"\004"	,	"(BC)"	,
	"(C)"	,	"(DE)"	,	"(HL)"	,	"\005"	,
	"\006"	,	"(SP)"	,	"A"	,	"AF"	,
	"AF'"	,	"B"	,	"BC"	,	"C"	,
	"D"	,	"DE"	,	"E"	,	"H"	,
	"HL"	,	"I"	,	"I%"	,	"L"	,
	"NC"	,	"NS"	,	"NV"	,	"NZ"	,
	"R"	,	"S"	,	"SP"	,	"V"	,
	"%H"	,	"%L"	,	"Z"	,	"0"	,
	"1"	,	"2"	,	"\007"	,	"\010"	,
};

typedef const char Z80_DASM_DICT[256][4]; // abridged dictionnaries made of the above lexical components
Z80_DASM_DICT z80_dasm_table= // unused slots contain zeroes (implicitly meaning Z80_DASM_NULL)
{
	// 0x00-0x0F
	{Z80_DASM_NOP	},
	{Z80_DASM_LD	,Z80_DASM_BC	,Z80_DASM_WORD	},
	{Z80_DASM_LD	,Z80_DASM_P_BC	,Z80_DASM_A	},
	{Z80_DASM_INC	,Z80_DASM_BC	},
	{Z80_DASM_INC	,Z80_DASM_B	},
	{Z80_DASM_DEC	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_BYTE	},
	{Z80_DASM_RLCA	},
	{Z80_DASM_EX	,Z80_DASM_AF	,Z80_DASM_AF2	},
	{Z80_DASM_ADD	,Z80_DASM_HL	,Z80_DASM_BC	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_P_BC	},
	{Z80_DASM_DEC	,Z80_DASM_BC	},
	{Z80_DASM_INC	,Z80_DASM_C	},
	{Z80_DASM_DEC	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_BYTE	},
	{Z80_DASM_RRCA	},
	// 0x10-0x1F
	{Z80_DASM_DJNZ	,Z80_DASM_R_PC	},
	{Z80_DASM_LD	,Z80_DASM_DE	,Z80_DASM_WORD	},
	{Z80_DASM_LD	,Z80_DASM_P_DE	,Z80_DASM_A	},
	{Z80_DASM_INC	,Z80_DASM_DE	},
	{Z80_DASM_INC	,Z80_DASM_D	},
	{Z80_DASM_DEC	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_BYTE	},
	{Z80_DASM_RLA	},
	{Z80_DASM_JR	,Z80_DASM_R_PC	},
	{Z80_DASM_ADD	,Z80_DASM_HL	,Z80_DASM_DE	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_P_DE	},
	{Z80_DASM_DEC	,Z80_DASM_DE	},
	{Z80_DASM_INC	,Z80_DASM_E	},
	{Z80_DASM_DEC	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_BYTE	},
	{Z80_DASM_RRA	},
	// 0x20-0x2F
	{Z80_DASM_JR	,Z80_DASM_NZ	,Z80_DASM_R_PC	},
	{Z80_DASM_LD	,Z80_DASM_HL	,Z80_DASM_WORD	},
	{Z80_DASM_LD	,Z80_DASM_P_W_	,Z80_DASM_HL	},
	{Z80_DASM_INC	,Z80_DASM_HL	},
	{Z80_DASM_INC	,Z80_DASM_H	},
	{Z80_DASM_DEC	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_BYTE	},
	{Z80_DASM_DAA	},
	{Z80_DASM_JR	,Z80_DASM_Z	,Z80_DASM_R_PC	},
	{Z80_DASM_ADD	,Z80_DASM_HL	,Z80_DASM_HL	},
	{Z80_DASM_LD	,Z80_DASM_HL	,Z80_DASM_P_W_	},
	{Z80_DASM_DEC	,Z80_DASM_HL	},
	{Z80_DASM_INC	,Z80_DASM_L	},
	{Z80_DASM_DEC	,Z80_DASM_L	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_BYTE	},
	{Z80_DASM_CPL	},
	// 0x30-0x3F
	{Z80_DASM_JR	,Z80_DASM_NC	,Z80_DASM_R_PC	},
	{Z80_DASM_LD	,Z80_DASM_SP	,Z80_DASM_WORD	},
	{Z80_DASM_LD	,Z80_DASM_P_W_	,Z80_DASM_A	},
	{Z80_DASM_INC	,Z80_DASM_SP	},
	{Z80_DASM_INC	,Z80_DASM_P_HL	},
	{Z80_DASM_DEC	,Z80_DASM_P_HL	},
	{Z80_DASM_LD	,Z80_DASM_P_HL	,Z80_DASM_BYTE	},
	{Z80_DASM_SCF	},
	{Z80_DASM_JR	,Z80_DASM_C	,Z80_DASM_R_PC	},
	{Z80_DASM_ADD	,Z80_DASM_HL	,Z80_DASM_SP	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_P_W_	},
	{Z80_DASM_DEC	,Z80_DASM_SP	},
	{Z80_DASM_INC	,Z80_DASM_A	},
	{Z80_DASM_DEC	,Z80_DASM_A	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_BYTE	},
	{Z80_DASM_CCF	},
	// 0x40-0x4F
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_L	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_P_HL	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_A	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_L	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_P_HL	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_A	},
	// 0x50-0x5F
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_L	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_P_HL	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_A	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_L	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_P_HL	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_A	},
	// 0x60-0x6F
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_L	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_P_HL	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_A	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_L	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_P_HL	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_A	},
	// 0x70-0x7F
	{Z80_DASM_LD	,Z80_DASM_P_HL	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_P_HL	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_P_HL	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_P_HL	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_P_HL	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_P_HL	,Z80_DASM_L	},
	{Z80_DASM_HALT	},
	{Z80_DASM_LD	,Z80_DASM_P_HL	,Z80_DASM_A	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_L	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_P_HL	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_A	},
	// 0x80-0x8F
	{Z80_DASM_ADD	,Z80_DASM_B	},
	{Z80_DASM_ADD	,Z80_DASM_C	},
	{Z80_DASM_ADD	,Z80_DASM_D	},
	{Z80_DASM_ADD	,Z80_DASM_E	},
	{Z80_DASM_ADD	,Z80_DASM_H	},
	{Z80_DASM_ADD	,Z80_DASM_L	},
	{Z80_DASM_ADD	,Z80_DASM_P_HL	},
	{Z80_DASM_ADD	,Z80_DASM_A	},
	{Z80_DASM_ADC	,Z80_DASM_B	},
	{Z80_DASM_ADC	,Z80_DASM_C	},
	{Z80_DASM_ADC	,Z80_DASM_D	},
	{Z80_DASM_ADC	,Z80_DASM_E	},
	{Z80_DASM_ADC	,Z80_DASM_H	},
	{Z80_DASM_ADC	,Z80_DASM_L	},
	{Z80_DASM_ADC	,Z80_DASM_P_HL	},
	{Z80_DASM_ADC	,Z80_DASM_A	},
	// 0x90-0x9F
	{Z80_DASM_SUB	,Z80_DASM_B	},
	{Z80_DASM_SUB	,Z80_DASM_C	},
	{Z80_DASM_SUB	,Z80_DASM_D	},
	{Z80_DASM_SUB	,Z80_DASM_E	},
	{Z80_DASM_SUB	,Z80_DASM_H	},
	{Z80_DASM_SUB	,Z80_DASM_L	},
	{Z80_DASM_SUB	,Z80_DASM_P_HL	},
	{Z80_DASM_SUB	,Z80_DASM_A	},
	{Z80_DASM_SBC	,Z80_DASM_B	},
	{Z80_DASM_SBC	,Z80_DASM_C	},
	{Z80_DASM_SBC	,Z80_DASM_D	},
	{Z80_DASM_SBC	,Z80_DASM_E	},
	{Z80_DASM_SBC	,Z80_DASM_H	},
	{Z80_DASM_SBC	,Z80_DASM_L	},
	{Z80_DASM_SBC	,Z80_DASM_P_HL	},
	{Z80_DASM_SBC	,Z80_DASM_A	},
	// 0xA0-0xAF
	{Z80_DASM_AND	,Z80_DASM_B	},
	{Z80_DASM_AND	,Z80_DASM_C	},
	{Z80_DASM_AND	,Z80_DASM_D	},
	{Z80_DASM_AND	,Z80_DASM_E	},
	{Z80_DASM_AND	,Z80_DASM_H	},
	{Z80_DASM_AND	,Z80_DASM_L	},
	{Z80_DASM_AND	,Z80_DASM_P_HL	},
	{Z80_DASM_AND	,Z80_DASM_A	},
	{Z80_DASM_XOR	,Z80_DASM_B	},
	{Z80_DASM_XOR	,Z80_DASM_C	},
	{Z80_DASM_XOR	,Z80_DASM_D	},
	{Z80_DASM_XOR	,Z80_DASM_E	},
	{Z80_DASM_XOR	,Z80_DASM_H	},
	{Z80_DASM_XOR	,Z80_DASM_L	},
	{Z80_DASM_XOR	,Z80_DASM_P_HL	},
	{Z80_DASM_XOR	,Z80_DASM_A	},
	// 0xB0-0xBF
	{Z80_DASM_OR	,Z80_DASM_B	},
	{Z80_DASM_OR	,Z80_DASM_C	},
	{Z80_DASM_OR	,Z80_DASM_D	},
	{Z80_DASM_OR	,Z80_DASM_E	},
	{Z80_DASM_OR	,Z80_DASM_H	},
	{Z80_DASM_OR	,Z80_DASM_L	},
	{Z80_DASM_OR	,Z80_DASM_P_HL	},
	{Z80_DASM_OR	,Z80_DASM_A	},
	{Z80_DASM_CP	,Z80_DASM_B	},
	{Z80_DASM_CP	,Z80_DASM_C	},
	{Z80_DASM_CP	,Z80_DASM_D	},
	{Z80_DASM_CP	,Z80_DASM_E	},
	{Z80_DASM_CP	,Z80_DASM_H	},
	{Z80_DASM_CP	,Z80_DASM_L	},
	{Z80_DASM_CP	,Z80_DASM_P_HL	},
	{Z80_DASM_CP	,Z80_DASM_A	},
	// 0xC0-0xCF
	{Z80_DASM_RET	,Z80_DASM_NZ	},
	{Z80_DASM_POP	,Z80_DASM_BC	},
	{Z80_DASM_JP	,Z80_DASM_NZ	,Z80_DASM_WORD	},
	{Z80_DASM_JP	,Z80_DASM_WORD	},
	{Z80_DASM_CALL	,Z80_DASM_NZ	,Z80_DASM_WORD	},
	{Z80_DASM_PUSH	,Z80_DASM_BC	},
	{Z80_DASM_ADD	,Z80_DASM_BYTE	},
	{Z80_DASM_RST	,Z80_DASM_DIV8	},
	{Z80_DASM_RET	,Z80_DASM_Z	},
	{Z80_DASM_RET	},
	{Z80_DASM_JP	,Z80_DASM_Z	,Z80_DASM_WORD	},
	{0}, // PREFIX CB
	{Z80_DASM_CALL	,Z80_DASM_Z	,Z80_DASM_WORD	},
	{Z80_DASM_CALL	,Z80_DASM_WORD	},
	{Z80_DASM_ADC	,Z80_DASM_BYTE	},
	{Z80_DASM_RST	,Z80_DASM_DIV8	},
	// 0xD0-0xDF
	{Z80_DASM_RET	,Z80_DASM_NC	},
	{Z80_DASM_POP	,Z80_DASM_DE	},
	{Z80_DASM_JP	,Z80_DASM_NC	,Z80_DASM_WORD	},
	{Z80_DASM_OUT	,Z80_DASM_P_B_	,Z80_DASM_A	},
	{Z80_DASM_CALL	,Z80_DASM_NC	,Z80_DASM_WORD	},
	{Z80_DASM_PUSH	,Z80_DASM_DE	},
	{Z80_DASM_SUB	,Z80_DASM_BYTE	},
	{Z80_DASM_RST	,Z80_DASM_DIV8	},
	{Z80_DASM_RET	,Z80_DASM_C	},
	{Z80_DASM_EXX	},
	{Z80_DASM_JP	,Z80_DASM_C	,Z80_DASM_WORD	},
	{Z80_DASM_IN	,Z80_DASM_A	,Z80_DASM_P_B_	},
	{Z80_DASM_CALL	,Z80_DASM_C	,Z80_DASM_WORD	},
	{0}, // PREFIX DD
	{Z80_DASM_SBC	,Z80_DASM_BYTE	},
	{Z80_DASM_RST	,Z80_DASM_DIV8	},
	// 0xE0-0xEF
	{Z80_DASM_RET	,Z80_DASM_NV	},
	{Z80_DASM_POP	,Z80_DASM_HL	},
	{Z80_DASM_JP	,Z80_DASM_NV	,Z80_DASM_WORD	},
	{Z80_DASM_EX	,Z80_DASM_HL	,Z80_DASM_P_SP	},
	{Z80_DASM_CALL	,Z80_DASM_NV	,Z80_DASM_WORD	},
	{Z80_DASM_PUSH	,Z80_DASM_HL	},
	{Z80_DASM_AND	,Z80_DASM_BYTE	},
	{Z80_DASM_RST	,Z80_DASM_DIV8	},
	{Z80_DASM_RET	,Z80_DASM_V	},
	{Z80_DASM_JP	,Z80_DASM_HL	},
	{Z80_DASM_JP	,Z80_DASM_V	,Z80_DASM_WORD	},
	{Z80_DASM_EX	,Z80_DASM_DE	,Z80_DASM_HL	},
	{Z80_DASM_CALL	,Z80_DASM_V	,Z80_DASM_WORD	},
	{0}, // PREFIX ED
	{Z80_DASM_XOR	,Z80_DASM_BYTE	},
	{Z80_DASM_RST	,Z80_DASM_DIV8	},
	// 0xF0-0xFF
	{Z80_DASM_RET	,Z80_DASM_NS	},
	{Z80_DASM_POP	,Z80_DASM_AF	},
	{Z80_DASM_JP	,Z80_DASM_NS	,Z80_DASM_WORD	},
	{Z80_DASM_DI	},
	{Z80_DASM_CALL	,Z80_DASM_NS	,Z80_DASM_WORD	},
	{Z80_DASM_PUSH	,Z80_DASM_AF	},
	{Z80_DASM_OR	,Z80_DASM_BYTE	},
	{Z80_DASM_RST	,Z80_DASM_DIV8	},
	{Z80_DASM_RET	,Z80_DASM_S	},
	{Z80_DASM_LD	,Z80_DASM_SP	,Z80_DASM_HL	},
	{Z80_DASM_JP	,Z80_DASM_S	,Z80_DASM_WORD	},
	{Z80_DASM_EI	},
	{Z80_DASM_CALL	,Z80_DASM_S	,Z80_DASM_WORD	},
	{0}, // PREFIX FD
	{Z80_DASM_CP	,Z80_DASM_BYTE	},
	{Z80_DASM_RST	,Z80_DASM_DIV8	},
};
Z80_DASM_DICT z80_dasm_table_cb=
{
	// 0xCB00-0xCB0F
	{Z80_DASM_RLC	,Z80_DASM_B	},
	{Z80_DASM_RLC	,Z80_DASM_C	},
	{Z80_DASM_RLC	,Z80_DASM_D	},
	{Z80_DASM_RLC	,Z80_DASM_E	},
	{Z80_DASM_RLC	,Z80_DASM_H	},
	{Z80_DASM_RLC	,Z80_DASM_L	},
	{Z80_DASM_RLC	,Z80_DASM_P_HL	},
	{Z80_DASM_RLC	,Z80_DASM_A	},
	{Z80_DASM_RRC	,Z80_DASM_B	},
	{Z80_DASM_RRC	,Z80_DASM_C	},
	{Z80_DASM_RRC	,Z80_DASM_D	},
	{Z80_DASM_RRC	,Z80_DASM_E	},
	{Z80_DASM_RRC	,Z80_DASM_H	},
	{Z80_DASM_RRC	,Z80_DASM_L	},
	{Z80_DASM_RRC	,Z80_DASM_P_HL	},
	{Z80_DASM_RRC	,Z80_DASM_A	},
	// 0xCB10-0xCB1F
	{Z80_DASM_RL	,Z80_DASM_B	},
	{Z80_DASM_RL	,Z80_DASM_C	},
	{Z80_DASM_RL	,Z80_DASM_D	},
	{Z80_DASM_RL	,Z80_DASM_E	},
	{Z80_DASM_RL	,Z80_DASM_H	},
	{Z80_DASM_RL	,Z80_DASM_L	},
	{Z80_DASM_RL	,Z80_DASM_P_HL	},
	{Z80_DASM_RL	,Z80_DASM_A	},
	{Z80_DASM_RR	,Z80_DASM_B	},
	{Z80_DASM_RR	,Z80_DASM_C	},
	{Z80_DASM_RR	,Z80_DASM_D	},
	{Z80_DASM_RR	,Z80_DASM_E	},
	{Z80_DASM_RR	,Z80_DASM_H	},
	{Z80_DASM_RR	,Z80_DASM_L	},
	{Z80_DASM_RR	,Z80_DASM_P_HL	},
	{Z80_DASM_RR	,Z80_DASM_A	},
	// 0xCB20-0xCB2F
	{Z80_DASM_SLA	,Z80_DASM_B	},
	{Z80_DASM_SLA	,Z80_DASM_C	},
	{Z80_DASM_SLA	,Z80_DASM_D	},
	{Z80_DASM_SLA	,Z80_DASM_E	},
	{Z80_DASM_SLA	,Z80_DASM_H	},
	{Z80_DASM_SLA	,Z80_DASM_L	},
	{Z80_DASM_SLA	,Z80_DASM_P_HL	},
	{Z80_DASM_SLA	,Z80_DASM_A	},
	{Z80_DASM_SRA	,Z80_DASM_B	},
	{Z80_DASM_SRA	,Z80_DASM_C	},
	{Z80_DASM_SRA	,Z80_DASM_D	},
	{Z80_DASM_SRA	,Z80_DASM_E	},
	{Z80_DASM_SRA	,Z80_DASM_H	},
	{Z80_DASM_SRA	,Z80_DASM_L	},
	{Z80_DASM_SRA	,Z80_DASM_P_HL	},
	{Z80_DASM_SRA	,Z80_DASM_A	},
	// 0xCB30-0xCB3F
	{Z80_DASM_SLL	,Z80_DASM_B	},
	{Z80_DASM_SLL	,Z80_DASM_C	},
	{Z80_DASM_SLL	,Z80_DASM_D	},
	{Z80_DASM_SLL	,Z80_DASM_E	},
	{Z80_DASM_SLL	,Z80_DASM_H	},
	{Z80_DASM_SLL	,Z80_DASM_L	},
	{Z80_DASM_SLL	,Z80_DASM_P_HL	},
	{Z80_DASM_SLL	,Z80_DASM_A	},
	{Z80_DASM_SRL	,Z80_DASM_B	},
	{Z80_DASM_SRL	,Z80_DASM_C	},
	{Z80_DASM_SRL	,Z80_DASM_D	},
	{Z80_DASM_SRL	,Z80_DASM_E	},
	{Z80_DASM_SRL	,Z80_DASM_H	},
	{Z80_DASM_SRL	,Z80_DASM_L	},
	{Z80_DASM_SRL	,Z80_DASM_P_HL	},
	{Z80_DASM_SRL	,Z80_DASM_A	},
	// 0xCB40-0xCB4F
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCB50-0xCB5F
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCB60-0xCB6F
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCB70-0xCB7F
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCB80-0xCB8F
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCB90-0xCB9F
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCBA0-0xCBAF
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCBB0-0xCBBF
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCBC0-0xCBCF
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCBD0-0xCBDF
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCBE0-0xCBEF
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	},
	// 0xCBF0-0xCBFF
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_P_HL	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	},
};
Z80_DASM_DICT z80_dasm_table_ed=
{
	// 0xED00-0xED0F
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xED10-0xED1F
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xED20-0xED2F
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xED30-0xED3F
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xED40-0xED4F
	{Z80_DASM_IN	,Z80_DASM_B	,Z80_DASM_P_C	},
	{Z80_DASM_OUT	,Z80_DASM_P_C	,Z80_DASM_B	},
	{Z80_DASM_SBC	,Z80_DASM_HL	,Z80_DASM_BC	},
	{Z80_DASM_LD	,Z80_DASM_P_W_	,Z80_DASM_BC	},
	{Z80_DASM_NEG	},
	{Z80_DASM_RETN	},
	{Z80_DASM_IM	,Z80_DASM_0	},
	{Z80_DASM_LD	,Z80_DASM_I	,Z80_DASM_A	},
	{Z80_DASM_IN	,Z80_DASM_C	,Z80_DASM_P_C	},
	{Z80_DASM_OUT	,Z80_DASM_P_C	,Z80_DASM_C	},
	{Z80_DASM_ADC	,Z80_DASM_HL	,Z80_DASM_BC	},
	{Z80_DASM_LD	,Z80_DASM_BC	,Z80_DASM_P_W_	},
	{Z80_DASM_NEG	},
	{Z80_DASM_RETI	},
	{Z80_DASM_IM	,Z80_DASM_0	},
	{Z80_DASM_LD	,Z80_DASM_R	,Z80_DASM_A	},
	// 0xED50-0xED5F
	{Z80_DASM_IN	,Z80_DASM_D	,Z80_DASM_P_C	},
	{Z80_DASM_OUT	,Z80_DASM_P_C	,Z80_DASM_D	},
	{Z80_DASM_SBC	,Z80_DASM_HL	,Z80_DASM_DE	},
	{Z80_DASM_LD	,Z80_DASM_P_W_	,Z80_DASM_DE	},
	{Z80_DASM_NEG	},
	{Z80_DASM_RETN	},
	{Z80_DASM_IM	,Z80_DASM_1	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_I	},
	{Z80_DASM_IN	,Z80_DASM_E	,Z80_DASM_P_C	},
	{Z80_DASM_OUT	,Z80_DASM_P_C	,Z80_DASM_E	},
	{Z80_DASM_ADC	,Z80_DASM_HL	,Z80_DASM_DE	},
	{Z80_DASM_LD	,Z80_DASM_DE	,Z80_DASM_P_W_	},
	{Z80_DASM_NEG	},
	{Z80_DASM_RETI	},
	{Z80_DASM_IM	,Z80_DASM_2	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_R	},
	// 0xED60-0xED6F
	{Z80_DASM_IN	,Z80_DASM_H	,Z80_DASM_P_C	},
	{Z80_DASM_OUT	,Z80_DASM_P_C	,Z80_DASM_H	},
	{Z80_DASM_SBC	,Z80_DASM_HL	,Z80_DASM_HL	},
	{Z80_DASM_LD	,Z80_DASM_P_W_	,Z80_DASM_HL	},
	{Z80_DASM_NEG	},
	{Z80_DASM_RETN	},
	{Z80_DASM_IM	,Z80_DASM_0	},
	{Z80_DASM_RRD	},
	{Z80_DASM_IN	,Z80_DASM_L	,Z80_DASM_P_C	},
	{Z80_DASM_OUT	,Z80_DASM_P_C	,Z80_DASM_L	},
	{Z80_DASM_ADC	,Z80_DASM_HL	,Z80_DASM_HL	},
	{Z80_DASM_LD	,Z80_DASM_HL	,Z80_DASM_P_W_	},
	{Z80_DASM_NEG	},
	{Z80_DASM_RETI	},
	{Z80_DASM_IM	,Z80_DASM_0	},
	{Z80_DASM_RLD	},
	// 0xED70-0xED7F
	{Z80_DASM_IN	,Z80_DASM_P_C	},
	{Z80_DASM_OUT	,Z80_DASM_P_C	},
	{Z80_DASM_SBC	,Z80_DASM_HL	,Z80_DASM_SP	},
	{Z80_DASM_LD	,Z80_DASM_P_W_	,Z80_DASM_SP	},
	{Z80_DASM_NEG	},
	{Z80_DASM_RETN	},
	{Z80_DASM_IM	,Z80_DASM_1	},
	{Z80_DASM_LD	,Z80_DASM_I	,Z80_DASM_I	},
	{Z80_DASM_IN	,Z80_DASM_A	,Z80_DASM_P_C	},
	{Z80_DASM_OUT	,Z80_DASM_P_C	,Z80_DASM_A	},
	{Z80_DASM_ADC	,Z80_DASM_HL	,Z80_DASM_SP	},
	{Z80_DASM_LD	,Z80_DASM_SP	,Z80_DASM_P_W_	},
	{Z80_DASM_NEG	},
	{Z80_DASM_RETI	},
	{Z80_DASM_IM	,Z80_DASM_2	},
	{Z80_DASM_LD	,Z80_DASM_R	,Z80_DASM_R	},
	// 0xED80-0xED8F
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xED90-0xED9F
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xEDA0-0xEDAF
	{Z80_DASM_LDI	},
	{Z80_DASM_CPI	},
	{Z80_DASM_INI	},
	{Z80_DASM_OUTI	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_LDD	},
	{Z80_DASM_CPD	},
	{Z80_DASM_IND	},
	{Z80_DASM_OUTD	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xEDB0-0xEDBF
	{Z80_DASM_LDIR	},
	{Z80_DASM_CPIR	},
	{Z80_DASM_INIR	},
	{Z80_DASM_OTIR	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_LDDR	},
	{Z80_DASM_CPDR	},
	{Z80_DASM_INDR	},
	{Z80_DASM_OTDR	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xEDC0-0xEDCF
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xEDD0-0xEDDF
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xEDE0-0xEDEF
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	// 0xEDF0-0xEDFF
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
	{Z80_DASM_SKIP	},
};
Z80_DASM_DICT z80_dasm_table_xy= // IX and IY are handled together
{
	// 0xDD00-0xDD0F
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_ADD	,Z80_DASM_IX	,Z80_DASM_BC	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	// 0xDD10-0xDD1F
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_ADD	,Z80_DASM_IX	,Z80_DASM_DE	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	// 0xDD20-0xDD2F
	{Z80_DASM_BACK	},
	{Z80_DASM_LD	,Z80_DASM_IX	,Z80_DASM_WORD	},
	{Z80_DASM_LD	,Z80_DASM_P_W_	,Z80_DASM_IX	},
	{Z80_DASM_INC	,Z80_DASM_IX	},
	{Z80_DASM_INC	,Z80_DASM_XH	},
	{Z80_DASM_DEC	,Z80_DASM_XH	},
	{Z80_DASM_LD	,Z80_DASM_XH	,Z80_DASM_BYTE	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_ADD	,Z80_DASM_IX	,Z80_DASM_IX	},
	{Z80_DASM_LD	,Z80_DASM_IX	,Z80_DASM_P_W_	},
	{Z80_DASM_DEC	,Z80_DASM_IX	},
	{Z80_DASM_INC	,Z80_DASM_XL	},
	{Z80_DASM_DEC	,Z80_DASM_XL	},
	{Z80_DASM_LD	,Z80_DASM_XL	,Z80_DASM_BYTE	},
	{Z80_DASM_BACK	},
	// 0xDD30-0xDD3F
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_INC	,Z80_DASM_P_IX	},
	{Z80_DASM_DEC	,Z80_DASM_P_IX	},
	{Z80_DASM_LD	,Z80_DASM_P_IX	,Z80_DASM_BYTE	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_ADD	,Z80_DASM_IX	,Z80_DASM_SP	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	// 0xDD40-0xDD4F
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_XH	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_XL	},
	{Z80_DASM_LD	,Z80_DASM_B	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_XH	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_XL	},
	{Z80_DASM_LD	,Z80_DASM_C	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	// 0xDD50-0xDD5F
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_XH	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_XL	},
	{Z80_DASM_LD	,Z80_DASM_D	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_XH	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_XL	},
	{Z80_DASM_LD	,Z80_DASM_E	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	// 0xDD60-0xDD6F
	{Z80_DASM_LD	,Z80_DASM_XH	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_XH	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_XH	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_XH	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_XH	,Z80_DASM_XH	},
	{Z80_DASM_LD	,Z80_DASM_XH	,Z80_DASM_XL	},
	{Z80_DASM_LD	,Z80_DASM_H	,Z80_DASM_P_IX	},
	{Z80_DASM_LD	,Z80_DASM_XH	,Z80_DASM_A	},
	{Z80_DASM_LD	,Z80_DASM_XL	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_XL	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_XL	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_XL	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_XL	,Z80_DASM_XH	},
	{Z80_DASM_LD	,Z80_DASM_XL	,Z80_DASM_XL	},
	{Z80_DASM_LD	,Z80_DASM_L	,Z80_DASM_P_IX	},
	{Z80_DASM_LD	,Z80_DASM_XL	,Z80_DASM_A	},
	// 0xDD70-0xDD7F
	{Z80_DASM_LD	,Z80_DASM_P_IX	,Z80_DASM_B	},
	{Z80_DASM_LD	,Z80_DASM_P_IX	,Z80_DASM_C	},
	{Z80_DASM_LD	,Z80_DASM_P_IX	,Z80_DASM_D	},
	{Z80_DASM_LD	,Z80_DASM_P_IX	,Z80_DASM_E	},
	{Z80_DASM_LD	,Z80_DASM_P_IX	,Z80_DASM_H	},
	{Z80_DASM_LD	,Z80_DASM_P_IX	,Z80_DASM_L	},
	{Z80_DASM_BACK	},
	{Z80_DASM_LD	,Z80_DASM_P_IX	,Z80_DASM_A	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_XH	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_XL	},
	{Z80_DASM_LD	,Z80_DASM_A	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	// 0xDD80-0xDD8F
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_ADD	,Z80_DASM_XH	},
	{Z80_DASM_ADD	,Z80_DASM_XL	},
	{Z80_DASM_ADD	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_ADC	,Z80_DASM_XH	},
	{Z80_DASM_ADC	,Z80_DASM_XL	},
	{Z80_DASM_ADC	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	// 0xDD90-0xDD9F
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_SUB	,Z80_DASM_XH	},
	{Z80_DASM_SUB	,Z80_DASM_XL	},
	{Z80_DASM_SUB	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_SBC	,Z80_DASM_XH	},
	{Z80_DASM_SBC	,Z80_DASM_XL	},
	{Z80_DASM_SBC	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	// 0xDDA0-0xDDAF
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_AND	,Z80_DASM_XH	},
	{Z80_DASM_AND	,Z80_DASM_XL	},
	{Z80_DASM_AND	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_XOR	,Z80_DASM_XH	},
	{Z80_DASM_XOR	,Z80_DASM_XL	},
	{Z80_DASM_XOR	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	// 0xDDB0-0xDDBF
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_OR	,Z80_DASM_XH	},
	{Z80_DASM_OR	,Z80_DASM_XL	},
	{Z80_DASM_OR	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_CP	,Z80_DASM_XH	},
	{Z80_DASM_CP	,Z80_DASM_XL	},
	{Z80_DASM_CP	,Z80_DASM_P_IX	},
	{Z80_DASM_BACK	},
	// 0xDDC0-0xDDCF
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{0}, // PREFIX DDCB
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	// 0xDDD0-0xDDDF
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	// 0xDDE0-0xDDEF
	{Z80_DASM_BACK	},
	{Z80_DASM_POP	,Z80_DASM_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_EX	,Z80_DASM_IX	,Z80_DASM_P_SP	},
	{Z80_DASM_BACK	},
	{Z80_DASM_PUSH	,Z80_DASM_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_JP	,Z80_DASM_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	// 0xDDF0-0xDDFF
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_LD	,Z80_DASM_SP	,Z80_DASM_IX	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
	{Z80_DASM_BACK	},
};
Z80_DASM_DICT z80_dasm_table_xycb= // IX and IY are handled together
{
	// 0xDDCBXX00-0xDDCBXX0F
	{Z80_DASM_RLC	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RLC	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RLC	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RLC	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RLC	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RLC	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RLC	,Z80_DASM_Q_IX	},
	{Z80_DASM_RLC	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_RRC	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RRC	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RRC	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RRC	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RRC	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RRC	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RRC	,Z80_DASM_Q_IX	},
	{Z80_DASM_RRC	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX10-0xDDCBXX1F
	{Z80_DASM_RL	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RL	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RL	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RL	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RL	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RL	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RL	,Z80_DASM_Q_IX	},
	{Z80_DASM_RL	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_RR	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RR	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RR	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RR	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RR	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RR	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RR	,Z80_DASM_Q_IX	},
	{Z80_DASM_RR	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX20-0xDDCBXX2F
	{Z80_DASM_SLA	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLA	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLA	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLA	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLA	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLA	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLA	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLA	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRA	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRA	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRA	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRA	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRA	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRA	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRA	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRA	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX30-0xDDCBXX3F
	{Z80_DASM_SLL	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLL	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLL	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLL	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLL	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLL	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLL	,Z80_DASM_Q_IX	},
	{Z80_DASM_SLL	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRL	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRL	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRL	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRL	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRL	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRL	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRL	,Z80_DASM_Q_IX	},
	{Z80_DASM_SRL	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX40-0xDDCBXX4F
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX50-0xDDCBXX5F
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX60-0xDDCBXX6F
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX70-0xDDCBXX7F
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_BIT	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX80-0xDDCBXX8F
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXX90-0xDDCBXX9F
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXXA0-0xDDCBXXAF
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXXB0-0xDDCBXXBF
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_RES	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXXC0-0xDDCBXXCF
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXXD0-0xDDCBXXDF
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXXE0-0xDDCBXXEF
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	// 0xDDCBXXF0-0xDDCBXXFF
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_B	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_C	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_D	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_E	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_H	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_L	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_Q_IX	},
	{Z80_DASM_SET	,Z80_DASM_DIV8	,Z80_DASM_A	,Z80_DASM_Q_IX	},
};

WORD z80_dasm(char *r,WORD m) // disassembles instruction at address `m` into string `r`: returns next instruction's address
{
	char *t=&r[15]; // format: "0123: DD364567 LD (IX+$45),$67"
	Z80_DASM_DICT *z80_dasm_table_o=&z80_dasm_table;
	BYTE i,o,xy,z; WORD p,n=m;
	switch (o=(PEEK(m))) // detect prefixes and select disassembly table
	{
		case 0xDD:
		case 0xFD:
			z80_dasm_table_o=&z80_dasm_table_xy;
			xy=o==0xDD?'X':'Y';
			++m; o=(PEEK(m));
			if (o==0xCB)
			{
				z80_dasm_table_o=&z80_dasm_table_xycb;
				++m; z=PEEK(m); // XYCB SUBSET is special:
				++m; o=PEEK(m); // RANGE goes before OPERATION
			}
			break;
		case 0xCB:
			z80_dasm_table_o=&z80_dasm_table_cb;
			++m; o=(PEEK(m));
			break;
		case 0xED:
			z80_dasm_table_o=&z80_dasm_table_ed;
			++m; o=(PEEK(m));
			break;
	}
	for (i=0;i<4;++i) // expand the dictionnary entries into a string
	{
		BYTE j=(*z80_dasm_table_o)[o][i];
		const char *s=z80_dasm_lex[j];
		if (j==Z80_DASM_BACK)
			--m;
		switch (*s)
		{
			case '\001': // Z80_DASM_BYTE
				++m; p=PEEK(m);
				t+=sprintf(t,"$%02X,",p);
				break;
			case '\002': // Z80_DASM_WORD
				++m; p=PEEK(m);
				++m; p+=PEEK(m)<<8;
				t+=sprintf(t,"$%04X,",p);
				break;
			case '\003': // Z80_DASM_P_B_
				++m; p=PEEK(m);
				t+=sprintf(t,"($%02X),",p);
				break;
			case '\004': // Z80_DASM_P_W_
				++m; p=PEEK(m);
				++m; p+=PEEK(m)<<8;
				t+=sprintf(t,"($%04X),",p);
				break;
			case '\005': // Z80_DASM_P_IX
				++m; z=PEEK(m); // RANGE goes after OPERATION
				// no `break`!
			case '\006': // Z80_DASM_Q_IX
				t+=sprintf(t,"(I%c%c$%02X),",xy,z&0x80?'-':'+',z&0x80?256-z:z);
				break;
			case '\007': // Z80_DASM_DIV8
				*t++='0'+((o/8)&7);
				*t++=',';
				break;
			case '\010': // Z80_DASM_R_PC
				++m; p=m+1+(signed char)PEEK(m);
				t+=sprintf(t,"$%04X,",p);
				// no `break`!
			case 0: break;
			default: // words
				while (j=*s++)
					*t++=j=='%'?xy:j;
				*t++=i?',':' '; // separator
				break;
		}
	}
	*--t=0; // remove last separator
	++m;
	// generate PC and byte dump
	t=r+sprintf(r,"%04X: ",n);
	while (n!=m)
	{
		t+=sprintf(t,"%02X",PEEK(n));
		++n;
	}
	memset(t,' ',&r[15]-t); // pad the string
	return m;
}

// Z80 emulator/debugger -------------------------------------------- //

#define PEEKW(x) (PEEK((WORD)(x))+(PEEK((WORD)(x+1))<<8))
#define POKEW(x,w) (POKE((WORD)(x))=(w),POKE((WORD)(x+1))=(w)>>8)
#ifdef DEBUG
char *z80_debug_scan(WORD *w,char *s)
{
	while (*s==' ')
		++s;
	char *t=s;
	if (*s>' ')
		*w=strtol(s,&s,16);
	return (t==s)?NULL:s; // no valid number = NULL!
}
WORD z80_debug_dump(char *t,WORD m)
{
	BYTE i,b;
	t+=sprintf(t,"%04X: ",m);
	for (i=0;i<16;++i)
		t+=sprintf(t,"%02X ",PEEK(m)),++m; // hexadecimal values
	m-=i;
	for (i=0;i<16;++i)
		b=PEEK(m),*t++=(b>=32&&b<127)?b:'.',++m; // printable ASCII characters
	*t=0;
	return m;
}
#endif

// macros are handier than typing the same snippets of code a million times
#define Z80_INC_R ++r7
#define Z80_DEC_R --r7
#define Z80_GET_R8 ((z80_ir.b.l&0x80)+(r7&0x7F)) // rebuild R from R7
#define Z80_FETCH Z80_PEEKX(z80_pc.w)
#define Z80_RD_PC Z80_PEEK(z80_pc.w)
#define Z80_WZ_PC z80_wz=Z80_RD_PC; ++z80_pc.w; z80_wz+=Z80_RD_PC<<8 // read WZ from WORD[PC++]
#define Z80_RD_HL BYTE b=Z80_PEEK(z80_hl.w)
#define Z80_WR_HL Z80_POKE(z80_hl.w,b)
#define Z80_WZ_XY z80_wz=xy->w+(signed char)(Z80_RD_PC); ++z80_pc.w // calculate WZ and XY from BYTE[PC++], without additional delays
#define Z80_WZ_XY_1X(x) z80_wz=xy->w+(signed char)(Z80_RD_PC); Z80_MREQ_1X(x,z80_pc.w); ++z80_pc.w // calculate WZ and XY from BYTE[PC++]
#define Z80_RD_WZ BYTE b=Z80_PEEK(z80_wz)
#define Z80_WR_WZ Z80_POKE(z80_wz,b)
#define Z80_LD2(x) x.l=Z80_RD_PC; ++z80_pc.w; x.h=Z80_RD_PC; ++z80_pc.w // load r16 from WORD[PC++]
#define Z80_RD2(x) Z80_WZ_PC; ++z80_pc.w; x.l=Z80_PEEK1(z80_wz); ++z80_wz; x.h=Z80_PEEK2(z80_wz) // load r16 from [WZ]
#define Z80_WR2(x) Z80_WZ_PC; ++z80_pc.w; Z80_POKE1(z80_wz,x.l); ++z80_wz; Z80_POKE2(z80_wz,x.h) // write r16 to [WZ]
#define Z80_EXX2(x,y) do{ int w=x; x=y; y=w; }while(0)
#define Z80_INC1(x) z80_af.b.l=(z80_af.b.l&0x01)+z80_flags_inc[++x]
#define Z80_DEC1(x) z80_af.b.l=(z80_af.b.l&0x01)+z80_flags_dec[--x]
#define Z80_ADD2(x,y) do{ DWORD z=x+y; z80_af.b.l=(z>>16)+((z>>8)&0x28)+(((z^x^y)>>8)&0x10)+(z80_af.b.l&0xC4); z80_wz=x+1; x=z; Z80_WAIT(7); }while(0)
#define Z80_ADC2(x) do{ DWORD z=z80_hl.w+x.w+(z80_af.b.l&1); z80_af.b.l=(z>>16)+(((z80_hl.w^z^x.w)>>8)&0x10)+(((WORD)z)?((z>>8)&0xA8):0x40)+((((x.w^~z80_hl.w)&(x.w^z))>>13)&4); z80_wz=z80_hl.w+1; z80_hl.w=z; Z80_WAIT(7); }while(0)
#define Z80_SBC2(x) do{ DWORD z=z80_hl.w-x.w-(z80_af.b.l&1); z80_af.b.l=2+((z>>16)&1)+(((z80_hl.w^z^x.w)>>8)&0x10)+(((WORD)z)?((z>>8)&0xA8):0x40)+((((x.w^z80_hl.w)&(z80_hl.w^z))>>13)&4); z80_wz=z80_hl.w+1; z80_hl.w=z; Z80_WAIT(7); }while(0)
#define Z80_ADDC(x,y) do{ DWORD z=z80_af.b.h+x+y; z80_af.b.l=z80_flags_sgn[(BYTE)z]+z80_flags_add[(z^z80_af.b.h^x)]; z80_af.b.h=z; }while(0) // ADD safely stays within [0,511]
#define Z80_SUBC(x,y) do{ DWORD z=z80_af.b.h-x-y; z80_af.b.l=z80_flags_sgn[(BYTE)z]+z80_flags_sub[(z^z80_af.b.h^x)&511]; z80_af.b.h=z; }while(0) // SUB can fall beyond [0,511]
#define Z80_ADD1(x) Z80_ADDC(x,0)
#define Z80_ADC1(x) Z80_ADDC(x,(z80_af.b.l&1))
#define Z80_SUB1(x) Z80_SUBC(x,0)
#define Z80_SBC1(x) Z80_SUBC(x,(z80_af.b.l&1))
#define Z80_AND1(x) z80_af.b.l=z80_flags_and[z80_af.b.h=z80_af.b.h&(x)]
#define Z80_XOR1(x) z80_af.b.l=z80_flags_xor[z80_af.b.h=z80_af.b.h^(x)]
#define Z80_OR1(x) z80_af.b.l=z80_flags_xor[z80_af.b.h=z80_af.b.h|(x)]
#define Z80_CP1(x) do{ DWORD z=z80_af.b.h-x; z80_af.b.l=(z80_flags_sgn[(BYTE)z]&0xD7)+z80_flags_sub[(z^z80_af.b.h^x)&511]+(x&0x28); }while(0) // unlike SUB, 1.- A intact, 2.- flags 3+5 from argument
#define Z80_RET2 z80_wz=Z80_PEEK(z80_sp.w); ++z80_sp.w; z80_pc.w=z80_wz+=Z80_PEEK(z80_sp.w)<<8; if (++z80_sp.w>z80_break_stack) { z80_break_stack=0xFFFF; z80_debug_reset(); session_signal|=SESSION_SIGNAL_DEBUG; _t_=0; } // break!
#define Z80_POP2(x) x.l=Z80_PEEK(z80_sp.w); ++z80_sp.w; x.h=Z80_PEEK(z80_sp.w); ++z80_sp.w
#define Z80_PUSH2(x) --z80_sp.w; Z80_POKE1(z80_sp.w,x.h); --z80_sp.w; Z80_POKE2(z80_sp.w,x.l)
#define Z80_CALL2 --z80_sp.w; Z80_POKE0(z80_sp.w,z80_pc.w>>8); --z80_sp.w; Z80_POKE0(z80_sp.w,z80_pc.w); z80_pc.w=z80_wz
#define Z80_RLC1(x) x=(x<<1)+(x>>7); z80_af.b.l=z80_flags_xor[x]+(x&1)
#define Z80_RRC1(x) x=(x>>1)+(x<<7); z80_af.b.l=z80_flags_xor[x]+((x>>7)&1)
#define Z80_RL1(x) do{ BYTE z=x>>7; z80_af.b.l=z80_flags_xor[x=(x<<1)+(z80_af.b.l&1)]+z; }while(0)
#define Z80_RR1(x) do{ BYTE z=x&1; z80_af.b.l=z80_flags_xor[x=(x>>1)+(z80_af.b.l<<7)]+z; }while(0)
#define Z80_SLA1(x) do{ BYTE z=x>>7; z80_af.b.l=z80_flags_xor[x=x<<1]+z; }while(0)
#define Z80_SRA1(x) do{ BYTE z=x&1; z80_af.b.l=z80_flags_xor[x=((signed char)x)>>1]+z; }while(0)
#define Z80_SLL1(x) do{ BYTE z=x>>7; z80_af.b.l=z80_flags_xor[x=(x<<1)+1]+z; }while(0)
#define Z80_SRL1(x) do{ BYTE z=x&1; z80_af.b.l=z80_flags_xor[x=x>>1]+z; }while(0)
#define Z80_BIT1(n,x,y) z80_af.b.l=(z80_flags_bit[x&(1<<n)]+(y&0x28))+(z80_af.b.l&1)
#define Z80_RES1(n,x) x&=~(1<<n)
#define Z80_SET1(n,x) x|=(1<<n)
#define Z80_IN2(x,y) z80_r7=r7; z80_wz=z80_bc.w; Z80_PRAE_RECV(z80_wz); z80_af.b.l=z80_flags_xor[x=Z80_RECV(z80_wz)]+(z80_af.b.l&1); Z80_POST_RECV(z80_wz); Z80_STRIDE_IO(y); r7=z80_r7; ++z80_wz
#define Z80_OUT2(x,y) z80_wz=z80_bc.w; Z80_PRAE_SEND(z80_wz); Z80_SEND(z80_wz,x); Z80_POST_SEND(z80_wz); Z80_STRIDE_IO(y); ++z80_wz
#define Z80_LDID2 do{ BYTE b=Z80_PEEK(z80_hl.w); Z80_POKE(z80_de.w,b); b+=z80_af.b.h; z80_af.b.l=(z80_af.b.l&0xC1)+(--z80_bc.w?0x04:0x00)+(b&8)+((b&2)<<4); }while(0)
#define Z80_CPID2 do{ BYTE b=Z80_PEEK(z80_hl.w); BYTE z=z80_af.b.h-b; z80_af.b.l=((z^z80_af.b.h^b)&0x10)+(z?(z&0x80):0x40)+(--z80_bc.w?0x06:0x02)+(z80_af.b.l&1); b=z-((z80_af.b.l>>4)&1); z80_af.b.l+=((b<<4)&0x20)+(b&8); }while(0) // ZS5H3V1-
#define Z80_INOTF(b,x) z80_af.b.l+=((z80_bc.b.l+(x)+b)>>8)&1 // common INID2+OTID2 flag logic
#define Z80_INID2(x,y) do{ Z80_WAIT(1); z80_wz=z80_bc.w; Z80_PRAE_RECV(z80_wz); BYTE b=Z80_RECV(z80_wz); Z80_POST_RECV(z80_wz); Z80_STRIDE_IO(y); Z80_POKE(z80_hl.w,b); z80_af.b.l=z80_flags_xor[--z80_bc.b.h]; Z80_INOTF(b,x); }while(0)
#define Z80_OTID2(x,y) do{ Z80_WAIT(1); z80_af.b.l=z80_flags_xor[--z80_bc.b.h]; BYTE b=Z80_PEEK(z80_hl.w); z80_wz=z80_bc.w; Z80_PRAE_SEND(z80_wz); Z80_SEND(z80_wz,b); Z80_POST_SEND(z80_wz); Z80_STRIDE_IO(y); Z80_INOTF(b,x); }while(0)

INLINE void z80_main(int _t_) // emulate the Z80 for `_t_` clock ticks
{
	#ifdef DEBUG
	static WORD breakpoint=0xFFFF;
	#endif
	int z80_t=0; // clock tick counter
	BYTE r7=z80_ir.b.l; // split R7+R8!
	Z80_AUXILIARY;
	do
	{
		#ifdef DEBUG
		if (session_signal&SESSION_SIGNAL_DEBUG)
		{
			if (feof(stdin))
			{
				session_signal&=~SESSION_SIGNAL_DEBUG;
				break;
			}
			session_please();
			static WORD m=0;
			int q=1,i; WORD w; char *s;
			z80_ir.b.l=Z80_GET_R8; // unify R7+R8!
			while (q)
			{
				char cmdline[STRMAX];
				w=z80_dasm(cmdline,z80_pc.w);
				z80_info();
				printf(
					"AF=%04X BC=%04X DE=%04X HL=%04X IX=%04X IY=%04X IR=%04X IM=%i %cI\n"
					"%-32s %04X: %04X %04X %04X %04X %04X\n>"
					,z80_af.w,z80_bc.w,z80_de.w,z80_hl.w,z80_ix.w,z80_iy.w,z80_ir.w,z80_imd,'D'+z80_iff.b.l
					,cmdline,z80_sp.w,PEEKW(z80_sp.w),PEEKW(z80_sp.w+2),PEEKW(z80_sp.w+4),PEEKW(z80_sp.w+6),PEEKW(z80_sp.w+8)
					);
				cmdline[0]=0;
				fgets(cmdline,STRMAX,stdin);
				switch (~32&cmdline[0])
				{
					case 'A':
						switch (~32&cmdline[1])
						{
							case 'F': // "AF"
								z80_debug_scan(&z80_af.w,&cmdline[2]);
								break;
						}
						break;
					case 'B':
						switch (~32&cmdline[1])
						{
							case 'C': // "BC"
								z80_debug_scan(&z80_bc.w,&cmdline[2]);
								break;
						}
						break;
					case 'D':
						switch (~32&cmdline[1])
						{
							case 'E': // "DE"
								z80_debug_scan(&z80_de.w,&cmdline[2]);
								break;
							default: // DISASSEMBLY
								z80_debug_scan(&m,&cmdline[1]);
								for (i=0;i<Z80_DEBUG_LEN;++i)
								{
									m=z80_dasm(cmdline,m);
									puts(cmdline);
								}
								break;
						}
						break;
					case 'H':
						switch (~32&cmdline[1])
						{
							case 'L': // "HL"
								z80_debug_scan(&z80_hl.w,&cmdline[2]);
								break;
							default: // QUICK HELP
								puts(
									"\nHelp..\n\th"
									"\nStep into\n\t<empty>"
									"\nSet register\n\tAF/BC/DE/HL/IR/IX/IY/SP <word>"
									"\nDisassemble [from address]\n\td [addr]\t"
									"\nJump to address\n\tj <addr>"
									"\nDump memory [from address]\n\tm [addr]"
									"\nPoke byte[s] into address\n\tp <addr> <byte..>"
									"\nSearch byte[s] from address\n\ts <addr> <byte..>"
									"\nRun [to address]\n\tr [addr]"
									"\nReturn from call\n\tu"
									"\nQuit debugger\n\tq"
								);
								break;
						}
						break;
					case 'I':
						switch (~32&cmdline[1])
						{
							case 'R': // "IR"
								z80_debug_scan(&z80_ir.w,&cmdline[2]);
								break;
							case 'X': // "IX"
								z80_debug_scan(&z80_ix.w,&cmdline[2]);
								break;
							case 'Y': // "IY"
								z80_debug_scan(&z80_iy.w,&cmdline[2]);
								break;
						}
						break;
					case 'J': // JUMP TO
						z80_debug_scan(&z80_pc.w,&cmdline[1]);
						break;
					case 'M': // MEMORY DUMP
						z80_debug_scan(&m,&cmdline[1]);
						for (i=0;i<Z80_DEBUG_LEN;++i)
						{
							m=z80_debug_dump(cmdline,m);
							puts(cmdline);
						}
						break;
					case 'P': // POKE
						s=z80_debug_scan(&m,&cmdline[1]);
						while (s=z80_debug_scan(&w,s))
							POKE(m)=w,++m;
						break;
					case 'R': // RUN TO / STEP OVER
						z80_debug_scan(&w,&cmdline[1]);
						z80_break_stack=0xFFFF; breakpoint=w;
						session_signal&=~SESSION_SIGNAL_DEBUG;
						q=0;
						break;
					case 'S':
						switch (~32&cmdline[1])
						{
							case 'P': // "SP"
								z80_debug_scan(&z80_sp.w,&cmdline[2]);
								break;
							default: // SEARCH
								if (s=z80_debug_scan(&m,&cmdline[1]))
								{
									i=m; m=0;
									unsigned char pattern[STRMAX];
									while (s=z80_debug_scan(&w,s))
										pattern[m++]=w;
									if (m)
									{
										int k=0;
										while (i<65536-m)
										{
											int j;
											for (j=0,w=i;j<m;++j,++w)
												if (PEEK(w)!=pattern[j])
													break;
											if (j>=m)
											{
												printf("%04X ",i);
												if (++k>=16*Z80_DEBUG_LEN)
												{
													printf("...");
													break; // too many finds!
												}
											}
											++i;
										}
										if (k)
											putchar('\n');
									}
								}
								break;
						}
						break;
					case 'U': // RETURN FROM..
						if (z80_sp.w>=0xFFFE) // fringe case?
							z80_break_stack=0xFFFF, breakpoint=w;
						else
							z80_break_stack=z80_sp.w, breakpoint=0xFFFF;
						session_signal&=~SESSION_SIGNAL_DEBUG;
						q=0;
						break;
					case 'Q': // QUIT
					case 0: // EOT
						session_signal&=~SESSION_SIGNAL_DEBUG;
					case 10: // LF
					case 13: // CR
						q=0; // STEP INTO
				}
			}
			r7=z80_ir.b.l; // split R7+R8!
		}
		#endif
		Z80_INC_R; // "Timing Tests 48k Spectrum" requires this!
		//if (z80_nmi) { z80_pc.w+=z80_halted; Z80_WAIT(7); Z80_STRIDE_X; Z80_STRIDE(0x3A); z80_wz=0x66; Z80_CALL2; z80_halted=z80_iff.b.l=z80_iff0=0; z80_nmi_ack(); } else
		if (z80_irq*z80_iff0) // z80_iff0 can be 0 or 1, but z80_irq can be any value
		{
			z80_pc.w+=z80_halted; // skip active HALT!
			if (z80_imd&2)
			{
				Z80_WAIT(7); Z80_STRIDE_X(0xE3); // IM 2 timing equals EX HL,(SP) : 19 T / 6 NOP
				z80_wz=(z80_ir.b.h<<8)+z80_irq_bus; // the address is built according to I and the bus
				z80_iff.b.l=Z80_PEEK(z80_wz);
				++z80_wz;
				z80_iff.b.h=Z80_PEEK(z80_wz);
				z80_wz=z80_iff.w;
			}
			else
			{
				Z80_WAIT(7); Z80_STRIDE_X(0x3A); // IM 0 and IM 1 timing equals LD A,($NNNN) : 13 T / 4 NOP (RST N is actually 11 T!)
				z80_wz=(z80_imd&1)?0x38:(z80_irq_bus&0x38); // IM 0 reads the address from the bus, `RST x` style; any opcode but a RST crashes anyway.
			}
			Z80_CALL2;
			z80_halted=z80_iff.w=z80_iff0=0;
			z80_irq_ack();
		}
		else
		{
			BYTE op=Z80_FETCH;
			Z80_STRIDE(op); ++z80_pc.w;
			Z80_STRIDE_0;
			z80_iff0=z80_iff.b.l; // EI delay
			switch (op)
			{
				// 0x00-0x3F
				case 0x00: // NOP
					break;
				case 0x01: // LD BC,$NNNN
					Z80_LD2(z80_bc.b);
					break;
				case 0x11: // LD DE,$NNNN
					Z80_LD2(z80_de.b);
					break;
				case 0x21: // LD HL,$NNNN
					Z80_LD2(z80_hl.b);
					break;
				case 0x31: // LD SP,$NNNN
					Z80_LD2(z80_sp.b);
					break;
				case 0x02: // LD (BC),A
					Z80_POKE(z80_bc.w,z80_af.b.h);
					z80_wz=((z80_bc.w+1)&0x00FF)|(z80_af.b.h<<8);
					break;
				case 0x12: // LD (DE),A
					Z80_POKE(z80_de.w,z80_af.b.h);
					z80_wz=((z80_de.w+1)&0x00FF)|(z80_af.b.h<<8);
					break;
				case 0x0A: // LD A,(BC)
					z80_af.b.h=Z80_PEEK(z80_bc.w);
					z80_wz=z80_bc.w+1;
					break;
				case 0x1A: // LD A,(DE)
					z80_af.b.h=Z80_PEEK(z80_de.w);
					z80_wz=z80_de.w+1;
					break;
				case 0x22: // LD ($NNNN),HL
					Z80_WR2(z80_hl.b);
					break;
				case 0x32: // LD ($NNNN),A
					Z80_WZ_PC; ++z80_pc.w;
					Z80_POKE(z80_wz,z80_af.b.h);
					z80_wz=((z80_wz+1)&0x00FF)|(z80_af.b.h<<8);
					break;
				case 0x2A: // LD HL,($NNNN)
					Z80_RD2(z80_hl.b);
					break;
				case 0x3A: // LD A,($NNNN)
					Z80_WZ_PC; ++z80_pc.w;
					z80_af.b.h=Z80_PEEK(z80_wz);
					++z80_wz;
					break;
				case 0x03: // INC BC
					++z80_bc.w; Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0x0B: // DEC BC
					--z80_bc.w; Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0x13: // INC DE
					++z80_de.w; Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0x1B: // DEC DE
					--z80_de.w; Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0x23: // INC HL
					++z80_hl.w; Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0x2B: // DEC HL
					--z80_hl.w; Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0x33: // INC SP
					++z80_sp.w; Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0x3B: // DEC SP
					--z80_sp.w; Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0x04: // INC B
					Z80_INC1(z80_bc.b.h);
					break;
				case 0x05: // DEC B
					Z80_DEC1(z80_bc.b.h);
					break;
				case 0x0C: // INC C
					Z80_INC1(z80_bc.b.l);
					break;
				case 0x0D: // DEC C
					Z80_DEC1(z80_bc.b.l);
					break;
				case 0x14: // INC D
					Z80_INC1(z80_de.b.h);
					break;
				case 0x15: // DEC D
					Z80_DEC1(z80_de.b.h);
					break;
				case 0x1C: // INC E
					Z80_INC1(z80_de.b.l);
					break;
				case 0x1D: // DEC E
					Z80_DEC1(z80_de.b.l);
					break;
				case 0x24: // INC H
					Z80_INC1(z80_hl.b.h);
					break;
				case 0x25: // DEC H
					Z80_DEC1(z80_hl.b.h);
					break;
				case 0x2C: // INC L
					Z80_INC1(z80_hl.b.l);
					break;
				case 0x2D: // DEC L
					Z80_DEC1(z80_hl.b.l);
					break;
				case 0x34: // INC (HL)
					{ Z80_RD_HL; Z80_INC1(b); Z80_IORQ_NEXT(1); Z80_WR_HL; }
					break;
				case 0x35: // DEC (HL)
					{ Z80_RD_HL; Z80_DEC1(b); Z80_IORQ_NEXT(1); Z80_WR_HL; }
					break;
				case 0x3C: // INC A
					Z80_INC1(z80_af.b.h);
					break;
				case 0x3D: // DEC A
					Z80_DEC1(z80_af.b.h);
					break;
				case 0x06: // LD B,$NN
					z80_bc.b.h=Z80_RD_PC; ++z80_pc.w;
					break;
				case 0x0E: // LD C,$NN
					z80_bc.b.l=Z80_RD_PC; ++z80_pc.w;
					break;
				case 0x16: // LD D,$NN
					z80_de.b.h=Z80_RD_PC; ++z80_pc.w;
					break;
				case 0x1E: // LD E,$NN
					z80_de.b.l=Z80_RD_PC; ++z80_pc.w;
					break;
				case 0x26: // LD H,$NN
					z80_hl.b.h=Z80_RD_PC; ++z80_pc.w;
					break;
				case 0x2E: // LD L,$NN
					z80_hl.b.l=Z80_RD_PC; ++z80_pc.w;
					break;
				case 0x36: // LD (HL),$NN
					{ BYTE b=Z80_RD_PC; Z80_POKE(z80_hl.w,b); ++z80_pc.w; }
					break;
				case 0x3E: // LD A,$NN
					z80_af.b.h=Z80_RD_PC; ++z80_pc.w;
					break;
				case 0x07: // RLCA
					z80_af.b.h=(z80_af.b.h<<1)+(z80_af.b.h>>7); z80_af.b.l=(z80_af.b.h&1)+(z80_af.b.h&0x28)+(z80_af.b.l&0xC4); // flags --503-0C
					break;
				case 0x0F: // RRCA
					z80_af.b.h=(z80_af.b.h>>1)+(z80_af.b.h<<7); z80_af.b.l=(z80_af.b.h>>7)+(z80_af.b.h&0x28)+(z80_af.b.l&0xC4); // flags --503-0C
					break;
				case 0x17: // RLA
					{ BYTE b=z80_af.b.h>>7; z80_af.b.h=(z80_af.b.h<<1)+(z80_af.b.l&1); z80_af.b.l=b+(z80_af.b.h&0x28)+(z80_af.b.l&0xC4); } // flags --503-0C
					break;
				case 0x1F: // RRA
					{ BYTE b=z80_af.b.h&1; z80_af.b.h=(z80_af.b.h>>1)+(z80_af.b.l<<7); z80_af.b.l=b+(z80_af.b.h&0x28)+(z80_af.b.l&0xC4); } // flags --503-0C
					break;
				case 0x08: // EX AF,AF'
					Z80_EXX2(z80_af.w,z80_af2.w);
					break;
				case 0x09: // ADD HL,BC
					Z80_ADD2(z80_hl.w,z80_bc.w);
					break;
				case 0x19: // ADD HL,DE
					Z80_ADD2(z80_hl.w,z80_de.w);
					break;
				case 0x29: // ADD HL,HL
					Z80_ADD2(z80_hl.w,z80_hl.w);
					break;
				case 0x39: // ADD HL,SP
					Z80_ADD2(z80_hl.w,z80_sp.w);
					break;
				case 0x10: // DJNZ $RR
					Z80_WAIT(1); z80_wz=(signed char)Z80_RD_PC; ++z80_pc.w;
					if (--z80_bc.b.h)
					{
						Z80_IORQ_1X_NEXT(5);
						z80_pc.w=z80_wz+=z80_pc.w;
						Z80_STRIDE(0x110);
					}
					break;
				case 0x18: // JR $RR
					z80_wz=(signed char)Z80_RD_PC;
					Z80_IORQ_1X_NEXT(5);
					z80_pc.w=z80_wz+=z80_pc.w+1;
					break;
				case 0x20: // JR NZ,$RR
					z80_wz=(signed char)Z80_RD_PC; ++z80_pc.w;
					if (!(z80_af.b.l&0x40))
					{
						Z80_IORQ_1X_NEXT(5);
						z80_pc.w=z80_wz+=z80_pc.w; Z80_STRIDE(0x120);
					}
					break;
				case 0x28: // JR Z,$RR
					z80_wz=(signed char)Z80_RD_PC; ++z80_pc.w;
					if (z80_af.b.l&0x40)
					{
						Z80_IORQ_1X_NEXT(5);
						z80_pc.w=z80_wz+=z80_pc.w; Z80_STRIDE(0x128);
					}
					break;
				case 0x30: // JR NC,$RR
					z80_wz=(signed char)Z80_RD_PC; ++z80_pc.w;
					if (!(z80_af.b.l&0x01))
					{
						Z80_IORQ_1X_NEXT(5);
						z80_pc.w=z80_wz+=z80_pc.w; Z80_STRIDE(0x130);
					}
					break;
				case 0x38: // JR C,$RR
					z80_wz=(signed char)Z80_RD_PC; ++z80_pc.w;
					if (z80_af.b.l&0x01)
					{
						Z80_IORQ_1X_NEXT(5);
						z80_pc.w=z80_wz+=z80_pc.w; Z80_STRIDE(0x138);
					}
					break;
				case 0x27: // DAA
					{
						BYTE x=z80_af.b.h,z=0,b=(((x&0x0F)>9)||(z80_af.b.l&0x10))?0x06:0;
						if ((x>0x99)||(z80_af.b.l&0x01))
							z=1,b+=0x60;
						if (z80_af.b.l&2)
							z80_af.b.h-=b;
						else
							z80_af.b.h+=b;
						z80_af.b.l=z80_flags_xor[z80_af.b.h]+((z80_af.b.h^x)&0x10)+(z80_af.b.l&2)+z;
					}
					break;
				case 0x2F: // CPL
					z80_af.b.l=(z80_af.b.l&0xC5)+((z80_af.b.h=~z80_af.b.h)&0x28)+0x12; // --5-3---
					break;
				case 0x37: // SCF
					z80_af.b.l=(z80_af.b.l&0xC4)+(z80_af.b.h&0x28)+1; // --5-3--1
					break;
				case 0x3F: // CCF
					z80_af.b.l=(z80_af.b.l&0xC4)+(z80_af.b.h&0x28)+((z80_af.b.l&1)<<4)+((~z80_af.b.l)&1); // --5H3--C
					break;
				// 0x40-0x7F
				case 0x41: // LD B,C
					z80_bc.b.h=z80_bc.b.l;
					break;
				case 0x42: // LD B,D
					z80_bc.b.h=z80_de.b.h;
					break;
				case 0x43: // LD B,E
					z80_bc.b.h=z80_de.b.l;
					break;
				case 0x44: // LD B,H
					z80_bc.b.h=z80_hl.b.h;
					break;
				case 0x45: // LD B,L
					z80_bc.b.h=z80_hl.b.l;
					break;
				case 0x46: // LD B,(HL)
					z80_bc.b.h=Z80_PEEK(z80_hl.w);
					break;
				case 0x47: // LD B,A
					z80_bc.b.h=z80_af.b.h;
					break;
				case 0x48: // LD C,B
					z80_bc.b.l=z80_bc.b.h;
					break;
				case 0x4A: // LD C,D
					z80_bc.b.l=z80_de.b.h;
					break;
				case 0x4B: // LD C,E
					z80_bc.b.l=z80_de.b.l;
					break;
				case 0x4C: // LD C,H
					z80_bc.b.l=z80_hl.b.h;
					break;
				case 0x4D: // LD C,L
					z80_bc.b.l=z80_hl.b.l;
					break;
				case 0x4E: // LD C,(HL)
					z80_bc.b.l=Z80_PEEK(z80_hl.w);
					break;
				case 0x4F: // LD C,A
					z80_bc.b.l=z80_af.b.h;
					break;
				case 0x50: // LD D,B
					z80_de.b.h=z80_bc.b.h;
					break;
				case 0x51: // LD D,C
					z80_de.b.h=z80_bc.b.l;
					break;
				case 0x53: // LD D,E
					z80_de.b.h=z80_de.b.l;
					break;
				case 0x54: // LD D,H
					z80_de.b.h=z80_hl.b.h;
					break;
				case 0x55: // LD D,L
					z80_de.b.h=z80_hl.b.l;
					break;
				case 0x56: // LD D,(HL)
					z80_de.b.h=Z80_PEEK(z80_hl.w);
					break;
				case 0x57: // LD D,A
					z80_de.b.h=z80_af.b.h;
					break;
				case 0x58: // LD E,B
					z80_de.b.l=z80_bc.b.h;
					break;
				case 0x59: // LD E,C
					z80_de.b.l=z80_bc.b.l;
					break;
				case 0x5A: // LD E,D
					z80_de.b.l=z80_de.b.h;
					break;
				case 0x5C: // LD E,H
					z80_de.b.l=z80_hl.b.h;
					break;
				case 0x5D: // LD E,L
					z80_de.b.l=z80_hl.b.l;
					break;
				case 0x5E: // LD E,(HL)
					z80_de.b.l=Z80_PEEK(z80_hl.w);
					break;
				case 0x5F: // LD E,A
					z80_de.b.l=z80_af.b.h;
					break;
				case 0x60: // LD H,B
					z80_hl.b.h=z80_bc.b.h;
					break;
				case 0x61: // LD H,C
					z80_hl.b.h=z80_bc.b.l;
					break;
				case 0x62: // LD H,D
					z80_hl.b.h=z80_de.b.h;
					break;
				case 0x63: // LD H,E
					z80_hl.b.h=z80_de.b.l;
					break;
				case 0x65: // LD H,L
					z80_hl.b.h=z80_hl.b.l;
					break;
				case 0x66: // LD H,(HL)
					z80_hl.b.h=Z80_PEEK(z80_hl.w);
					break;
				case 0x67: // LD H,A
					z80_hl.b.h=z80_af.b.h;
					break;
				case 0x68: // LD L,B
					z80_hl.b.l=z80_bc.b.h;
					break;
				case 0x69: // LD L,C
					z80_hl.b.l=z80_bc.b.l;
					break;
				case 0x6A: // LD L,D
					z80_hl.b.l=z80_de.b.h;
					break;
				case 0x6B: // LD L,E
					z80_hl.b.l=z80_de.b.l;
					break;
				case 0x6C: // LD L,H
					z80_hl.b.l=z80_hl.b.h;
					break;
				case 0x6E: // LD L,(HL)
					z80_hl.b.l=Z80_PEEK(z80_hl.w);
					break;
				case 0x6F: // LD L,A
					z80_hl.b.l=z80_af.b.h;
					break;
				case 0x70: // LD (HL),B
					Z80_POKE(z80_hl.w,z80_bc.b.h);
					break;
				case 0x71: // LD (HL),C
					Z80_POKE(z80_hl.w,z80_bc.b.l);
					break;
				case 0x72: // LD (HL),D
					Z80_POKE(z80_hl.w,z80_de.b.h);
					break;
				case 0x73: // LD (HL),E
					Z80_POKE(z80_hl.w,z80_de.b.l);
					break;
				case 0x74: // LD (HL),H
					Z80_POKE(z80_hl.w,z80_hl.b.h);
					break;
				case 0x75: // LD (HL),L
					Z80_POKE(z80_hl.w,z80_hl.b.l);
					break;
				case 0x77: // LD (HL),A
					Z80_POKE(z80_hl.w,z80_af.b.h);
					break;
				case 0x78: // LD A,B
					z80_af.b.h=z80_bc.b.h;
					break;
				case 0x79: // LD A,C
					z80_af.b.h=z80_bc.b.l;
					break;
				case 0x7A: // LD A,D
					z80_af.b.h=z80_de.b.h;
					break;
				case 0x7B: // LD A,E
					z80_af.b.h=z80_de.b.l;
					break;
				case 0x7C: // LD A,H
					z80_af.b.h=z80_hl.b.h;
					break;
				case 0x7D: // LD A,L
					z80_af.b.h=z80_hl.b.l;
					break;
				case 0x7E: // LD A,(HL)
					z80_af.b.h=Z80_PEEK(z80_hl.w);
				case 0x40: // LD B,B
					//z80_bc.b.h=z80_bc.b.h;
					//break;
				case 0x49: // LD C,C
					//z80_bc.b.l=z80_bc.b.l;
					//break;
				case 0x52: // LD D,D
					//z80_de.b.h=z80_de.b.h;
					//break;
				case 0x5B: // LD E,E
					//z80_de.b.l=z80_de.b.l;
					//break;
				case 0x64: // LD H,H
					//z80_hl.b.h=z80_hl.b.h;
					//break;
				case 0x6D: // LD L,L
					//z80_hl.b.l=z80_hl.b.l;
					//break;
				case 0x7F: // LD A,A
					//z80_af.b.h=z80_af.b.h;
					//break;
					break;
				case 0x76: // HALT
					if (z80_iff0==z80_iff.b.l&&_t_>z80_t) // optimisation!
					{
						int z=(_t_-z80_t)/Z80_STRIDE_HALT;
						z80_r7+=z;
						z80_t+=z*Z80_STRIDE_HALT;
					}
					--z80_pc.w,z80_halted=1;
					break;
				// 0x80-0xBF
				case 0x80: // ADD B
					Z80_ADD1(z80_bc.b.h);
					break;
				case 0x81: // ADD C
					Z80_ADD1(z80_bc.b.l);
					break;
				case 0x82: // ADD D
					Z80_ADD1(z80_de.b.h);
					break;
				case 0x83: // ADD E
					Z80_ADD1(z80_de.b.l);
					break;
				case 0x84: // ADD H
					Z80_ADD1(z80_hl.b.h);
					break;
				case 0x85: // ADD L
					Z80_ADD1(z80_hl.b.l);
					break;
				case 0x86: // ADD (HL)
					{ Z80_RD_HL; Z80_ADD1(b); }
					break;
				case 0x87: // ADD A
					Z80_ADD1(z80_af.b.h);
					break;
				case 0x88: // ADC B
					Z80_ADC1(z80_bc.b.h);
					break;
				case 0x89: // ADC C
					Z80_ADC1(z80_bc.b.l);
					break;
				case 0x8A: // ADC D
					Z80_ADC1(z80_de.b.h);
					break;
				case 0x8B: // ADC E
					Z80_ADC1(z80_de.b.l);
					break;
				case 0x8C: // ADC H
					Z80_ADC1(z80_hl.b.h);
					break;
				case 0x8D: // ADC L
					Z80_ADC1(z80_hl.b.l);
					break;
				case 0x8E: // ADC (HL)
					{ Z80_RD_HL; Z80_ADC1(b); }
					break;
				case 0x8F: // ADC A
					Z80_ADC1(z80_af.b.h);
					break;
				case 0x90: // SUB B
					Z80_SUB1(z80_bc.b.h);
					break;
				case 0x91: // SUB C
					Z80_SUB1(z80_bc.b.l);
					break;
				case 0x92: // SUB D
					Z80_SUB1(z80_de.b.h);
					break;
				case 0x93: // SUB E
					Z80_SUB1(z80_de.b.l);
					break;
				case 0x94: // SUB H
					Z80_SUB1(z80_hl.b.h);
					break;
				case 0x95: // SUB L
					Z80_SUB1(z80_hl.b.l);
					break;
				case 0x96: // SUB (HL)
					{ Z80_RD_HL; Z80_SUB1(b); }
					break;
				case 0x97: // SUB A
					Z80_SUB1(z80_af.b.h);
					break;
				case 0x98: // SBC B
					Z80_SBC1(z80_bc.b.h);
					break;
				case 0x99: // SBC C
					Z80_SBC1(z80_bc.b.l);
					break;
				case 0x9A: // SBC D
					Z80_SBC1(z80_de.b.h);
					break;
				case 0x9B: // SBC E
					Z80_SBC1(z80_de.b.l);
					break;
				case 0x9C: // SBC H
					Z80_SBC1(z80_hl.b.h);
					break;
				case 0x9D: // SBC L
					Z80_SBC1(z80_hl.b.l);
					break;
				case 0x9E: // SBC (HL)
					{ Z80_RD_HL; Z80_SBC1(b); }
					break;
				case 0x9F: // SBC A
					Z80_SBC1(z80_af.b.h);
					break;
				case 0xA0: // AND B
					Z80_AND1(z80_bc.b.h);
					break;
				case 0xA1: // AND C
					Z80_AND1(z80_bc.b.l);
					break;
				case 0xA2: // AND D
					Z80_AND1(z80_de.b.h);
					break;
				case 0xA3: // AND E
					Z80_AND1(z80_de.b.l);
					break;
				case 0xA4: // AND H
					Z80_AND1(z80_hl.b.h);
					break;
				case 0xA5: // AND L
					Z80_AND1(z80_hl.b.l);
					break;
				case 0xA6: // AND (HL)
					{ Z80_RD_HL; Z80_AND1(b); }
					break;
				case 0xA7: // AND A
					Z80_AND1(z80_af.b.h);
					break;
				case 0xA8: // XOR B
					Z80_XOR1(z80_bc.b.h);
					break;
				case 0xA9: // XOR C
					Z80_XOR1(z80_bc.b.l);
					break;
				case 0xAA: // XOR D
					Z80_XOR1(z80_de.b.h);
					break;
				case 0xAB: // XOR E
					Z80_XOR1(z80_de.b.l);
					break;
				case 0xAC: // XOR H
					Z80_XOR1(z80_hl.b.h);
					break;
				case 0xAD: // XOR L
					Z80_XOR1(z80_hl.b.l);
					break;
				case 0xAE: // XOR (HL)
					{ Z80_RD_HL; Z80_XOR1(b); }
					break;
				case 0xAF: // XOR A
					Z80_XOR1(z80_af.b.h);
					break;
				case 0xB0: // OR B
					Z80_OR1(z80_bc.b.h);
					break;
				case 0xB1: // OR C
					Z80_OR1(z80_bc.b.l);
					break;
				case 0xB2: // OR D
					Z80_OR1(z80_de.b.h);
					break;
				case 0xB3: // OR E
					Z80_OR1(z80_de.b.l);
					break;
				case 0xB4: // OR H
					Z80_OR1(z80_hl.b.h);
					break;
				case 0xB5: // OR L
					Z80_OR1(z80_hl.b.l);
					break;
				case 0xB6: // OR (HL)
					{ Z80_RD_HL; Z80_OR1(b); }
					break;
				case 0xB7: // OR A
					Z80_OR1(z80_af.b.h);
					break;
				case 0xB8: // CP B
					Z80_CP1(z80_bc.b.h);
					break;
				case 0xB9: // CP C
					Z80_CP1(z80_bc.b.l);
					break;
				case 0xBA: // CP D
					Z80_CP1(z80_de.b.h);
					break;
				case 0xBB: // CP E
					Z80_CP1(z80_de.b.l);
					break;
				case 0xBC: // CP H
					Z80_CP1(z80_hl.b.h);
					break;
				case 0xBD: // CP L
					Z80_CP1(z80_hl.b.l);
					break;
				case 0xBE: // CP (HL)
					{ Z80_RD_HL; Z80_CP1(b); }
					break;
				case 0xBF: // CP A
					Z80_CP1(z80_af.b.h);
					break;
				// 0xC0-0xFF
				case 0xC0: // RET NZ
					Z80_WAIT(1); if (!(z80_af.b.l&0x40))
					{
						Z80_RET2;
						Z80_STRIDE(0x1C0);
					}
					else
						Z80_STRIDE_1;
					break;
				case 0xC8: // RET Z
					Z80_WAIT(1); if (z80_af.b.l&0x40)
					{
						Z80_RET2;
						Z80_STRIDE(0x1C8);
					}
					else
						Z80_STRIDE_1;
					break;
				case 0xD0: // RET NC
					Z80_WAIT(1); if (!(z80_af.b.l&0x01))
					{
						Z80_RET2;
						Z80_STRIDE(0x1D0);
					}
					else
						Z80_STRIDE_1;
					break;
				case 0xD8: // RET C
					Z80_WAIT(1); if (z80_af.b.l&0x01)
					{
						Z80_RET2;
						Z80_STRIDE(0x1D8);
					}
					else
						Z80_STRIDE_1;
					break;
				case 0xE0: // RET NV
					Z80_WAIT(1); if (!(z80_af.b.l&0x04))
					{
						Z80_RET2;
						Z80_STRIDE(0x1E0);
					}
					else
						Z80_STRIDE_1;
					break;
				case 0xE8: // RET V
					Z80_WAIT(1); if (z80_af.b.l&0x04)
					{
						Z80_RET2;
						Z80_STRIDE(0x1E8);
					}
					else
						Z80_STRIDE_1;
					break;
				case 0xF0: // RET NS
					Z80_WAIT(1); if (!(z80_af.b.l&0x80))
					{
						Z80_RET2;
						Z80_STRIDE(0x1F0);
					}
					else
						Z80_STRIDE_1;
					break;
				case 0xF8: // RET S
					Z80_WAIT(1); if (z80_af.b.l&0x80)
					{
						Z80_RET2;
						Z80_STRIDE(0x1F8);
					}
					else
						Z80_STRIDE_1;
					break;
				case 0xC9: // RET
					Z80_RET2;
					break;
				case 0xC1: // POP BC
					Z80_POP2(z80_bc.b);
					break;
				case 0xC5: // PUSH BC
					Z80_WAIT(1);
					Z80_PUSH2(z80_bc.b);
					break;
				case 0xD1: // POP DE
					Z80_POP2(z80_de.b);
					break;
				case 0xD5: // PUSH DE
					Z80_WAIT(1);
					Z80_PUSH2(z80_de.b);
					break;
				case 0xE1: // POP HL
					Z80_POP2(z80_hl.b);
					break;
				case 0xE5: // PUSH HL
					Z80_WAIT(1);
					Z80_PUSH2(z80_hl.b);
					break;
				case 0xF1: // POP AF
					Z80_POP2(z80_af.b);
					break;
				case 0xF5: // PUSH AF
					Z80_WAIT(1);
					Z80_PUSH2(z80_af.b);
					break;
				case 0xC2: // JP NZ,$NNNN
					Z80_WZ_PC;
					if (!(z80_af.b.l&0x40))
						z80_pc.w=z80_wz;
					else
						++z80_pc.w;
					break;
				case 0xCA: // JP Z,$NNNN
					Z80_WZ_PC;
					if (z80_af.b.l&0x40)
						z80_pc.w=z80_wz;
					else
						++z80_pc.w;
					break;
				case 0xD2: // JP NC,$NNNN
					Z80_WZ_PC;
					if (!(z80_af.b.l&0x01))
						z80_pc.w=z80_wz;
					else
						++z80_pc.w;
					break;
				case 0xDA: // JP C,$NNNN
					Z80_WZ_PC;
					if (z80_af.b.l&0x01)
						z80_pc.w=z80_wz;
					else
						++z80_pc.w;
					break;
				case 0xE2: // JP NV,$NNNN
					Z80_WZ_PC;
					if (!(z80_af.b.l&0x04))
						z80_pc.w=z80_wz;
					else
						++z80_pc.w;
					break;
				case 0xEA: // JP V,$NNNN
					Z80_WZ_PC;
					if (z80_af.b.l&0x04)
						z80_pc.w=z80_wz;
					else
						++z80_pc.w;
					break;
				case 0xF2: // JP NS,$NNNN
					Z80_WZ_PC;
					if (!(z80_af.b.l&0x80))
						z80_pc.w=z80_wz;
					else
						++z80_pc.w;
					break;
				case 0xFA: // JP S,$NNNN
					Z80_WZ_PC;
					if (z80_af.b.l&0x80)
						z80_pc.w=z80_wz;
					else
						++z80_pc.w;
					break;
				case 0xC3: // JP $NNNN
					Z80_WZ_PC;
					z80_pc.w=z80_wz;
					break;
				case 0xC4: // CALL NZ,$NNNN
					Z80_WZ_PC; ++z80_pc.w;
					if (!(z80_af.b.l&0x40))
					{
						Z80_IORQ_NEXT(1); Z80_CALL2;
						Z80_STRIDE(0x1C4);
					}
					break;
				case 0xCC: // CALL Z,$NNNN
					Z80_WZ_PC; ++z80_pc.w;
					if (z80_af.b.l&0x40)
					{
						Z80_IORQ_NEXT(1); Z80_CALL2;
						Z80_STRIDE(0x1CC);
					}
					break;
				case 0xD4: // CALL NC,$NNNN
					Z80_WZ_PC; ++z80_pc.w;
					if (!(z80_af.b.l&0x01))
					{
						Z80_IORQ_NEXT(1); Z80_CALL2;
						Z80_STRIDE(0x1D4);
					}
					break;
				case 0xDC: // CALL C,$NNNN
					Z80_WZ_PC; ++z80_pc.w;
					if (z80_af.b.l&0x01)
					{
						Z80_IORQ_NEXT(1); Z80_CALL2;
						Z80_STRIDE(0x1DC);
					}
					break;
				case 0xE4: // CALL NV,$NNNN
					Z80_WZ_PC; ++z80_pc.w;
					if (!(z80_af.b.l&0x04))
					{
						Z80_IORQ_NEXT(1); Z80_CALL2;
						Z80_STRIDE(0x1E4);
					}
					break;
				case 0xEC: // CALL V,$NNNN
					Z80_WZ_PC; ++z80_pc.w;
					if (z80_af.b.l&0x04)
					{
						Z80_IORQ_NEXT(1); Z80_CALL2;
						Z80_STRIDE(0x1EC);
					}
					break;
				case 0xF4: // CALL NS,$NNNN
					Z80_WZ_PC; ++z80_pc.w;
					if (!(z80_af.b.l&0x80))
					{
						Z80_IORQ_NEXT(1); Z80_CALL2;
						Z80_STRIDE(0x1F4);
					}
					break;
				case 0xFC: // CALL S,$NNNN
					Z80_WZ_PC; ++z80_pc.w;
					if (z80_af.b.l&0x80)
					{
						Z80_IORQ_NEXT(1); Z80_CALL2;
						Z80_STRIDE(0x1FC);
					}
					break;
				case 0xCD: // CALL $NNNN
					Z80_WZ_PC; ++z80_pc.w;
					Z80_IORQ_NEXT(1); Z80_CALL2;
					break;
				case 0xC6: // ADD $NN
					{ BYTE b=Z80_RD_PC; ++z80_pc.w; Z80_ADD1(b); }
					break;
				case 0xCE: // ADC $NN
					{ BYTE b=Z80_RD_PC; ++z80_pc.w; Z80_ADC1(b); }
					break;
				case 0xD6: // SUB $NN
					{ BYTE b=Z80_RD_PC; ++z80_pc.w; Z80_SUB1(b); }
					break;
				case 0xDE: // SBC $NN
					{ BYTE b=Z80_RD_PC; ++z80_pc.w; Z80_SBC1(b); }
					break;
				case 0xE6: // AND $NN
					{ BYTE b=Z80_RD_PC; ++z80_pc.w; Z80_AND1(b); }
					break;
				case 0xEE: // XOR $NN
					{ BYTE b=Z80_RD_PC; ++z80_pc.w; Z80_XOR1(b); }
					break;
				case 0xF6: // OR $NN
					{ BYTE b=Z80_RD_PC; ++z80_pc.w; Z80_OR1(b); }
					break;
				case 0xFE: // CP $NN
					{ BYTE b=Z80_RD_PC; ++z80_pc.w; Z80_CP1(b); }
					break;
				case 0xDB: // IN A,($NN)
					z80_wz=(z80_af.b.h<<8)+Z80_RD_PC; ++z80_pc.w;
					Z80_PRAE_RECV(z80_wz);
					z80_af.b.h=Z80_RECV(z80_wz);
					Z80_POST_RECV(z80_wz);
					Z80_STRIDE_IO(0x1DB);
					++z80_wz;
					break;
				case 0xD3: // OUT ($NN),A
					z80_wz=(z80_af.b.h<<8)+Z80_RD_PC; ++z80_pc.w;
					Z80_PRAE_SEND(z80_wz);
					Z80_SEND(z80_wz,z80_af.b.h);
					Z80_POST_SEND(z80_wz);
					Z80_STRIDE_IO(0x1D3);
					z80_wz=((z80_wz+1)&0x00FF)|(z80_af.b.h<<8);
					break;
				case 0xD9: // EXX
					Z80_EXX2(z80_bc.w,z80_bc2.w);
					Z80_EXX2(z80_de.w,z80_de2.w);
					Z80_EXX2(z80_hl.w,z80_hl2.w);
					break;
				case 0xEB: // EX DE,HL
					Z80_EXX2(z80_de.w,z80_hl.w);
					break;
				case 0xE3: // EX HL,(SP)
					z80_wz=Z80_PEEK(z80_sp.w);
					++z80_sp.w;
					z80_wz+=Z80_PEEKX(z80_sp.w)<<8;
					Z80_POKE1(z80_sp.w,z80_hl.b.h);
					--z80_sp.w;
					Z80_POKE2(z80_sp.w,z80_hl.b.l);
					z80_hl.w=z80_wz;
					Z80_IORQ_1X_NEXT(2);
					Z80_STRIDE_1;
					break;
				case 0xE9: // JP HL
					z80_pc.w=z80_hl.w;
					break;
				case 0xF9: // LD SP,HL
					z80_sp.w=z80_hl.w;
					Z80_STRIDE_1;
					Z80_WAIT(2);
					break;
				case 0xF3: // DI
					z80_iff.w=z80_iff0=0; // disable interruptions at once
					break;
				case 0xFB: // EI
					z80_iff.w=0x0101; // enable interruptions one instruction later
					break;
				case 0xC7: // RST 0
				case 0xCF: // RST 1
				case 0xD7: // RST 2
				case 0xDF: // RST 3
				case 0xE7: // RST 4
				case 0xEF: // RST 5
				case 0xF7: // RST 6
				case 0xFF: // RST 7
					z80_wz=op&0x38;
					Z80_WAIT(1); Z80_CALL2;
					break;
				case 0xCB: // PREFIX: CB SUBSET
					Z80_INC_R; op=Z80_FETCH;
					Z80_STRIDE(op+0x400); ++z80_pc.w;
					// the CB set is extremely repetitive and thus worth abridging
					#define CASE_Z80_CB_OP1(xx,yy) \
						case xx+0: yy(z80_bc.b.h); break; case xx+1: yy(z80_bc.b.l); break; \
						case xx+2: yy(z80_de.b.h); break; case xx+3: yy(z80_de.b.l); break; \
						case xx+4: yy(z80_hl.b.h); break; case xx+5: yy(z80_hl.b.l); break; \
						case xx+7: yy(z80_af.b.h); break; case xx+6: { Z80_RD_HL; yy(b); Z80_IORQ_NEXT(1); Z80_WR_HL; } break
					#define CASE_Z80_CB_BIT(xx,yy) \
						case xx+0: Z80_BIT1(yy,z80_bc.b.h,z80_bc.b.h); break; case xx+1: Z80_BIT1(yy,z80_bc.b.l,z80_bc.b.l); break; \
						case xx+2: Z80_BIT1(yy,z80_de.b.h,z80_de.b.h); break; case xx+3: Z80_BIT1(yy,z80_de.b.l,z80_de.b.l); break; \
						case xx+4: Z80_BIT1(yy,z80_hl.b.h,z80_hl.b.h); break; case xx+5: Z80_BIT1(yy,z80_hl.b.l,z80_hl.b.l); break; \
						case xx+7: Z80_BIT1(yy,z80_af.b.h,z80_af.b.h); break; case xx+6: { Z80_RD_HL; Z80_BIT1(yy,b,(z80_wz>>8)); Z80_IORQ_NEXT(1); } break
					#define CASE_Z80_CB_OP2(xx,yy,zz) \
						case xx+0: yy(zz,z80_bc.b.h); break; case xx+1: yy(zz,z80_bc.b.l); break; \
						case xx+2: yy(zz,z80_de.b.h); break; case xx+3: yy(zz,z80_de.b.l); break; \
						case xx+4: yy(zz,z80_hl.b.h); break; case xx+5: yy(zz,z80_hl.b.l); break; \
						case xx+7: yy(zz,z80_af.b.h); break; case xx+6: { Z80_RD_HL; yy(zz,b); Z80_IORQ_NEXT(1); Z80_WR_HL; } break
					switch (op)
					{
						// 0xCB00-0xCB3F
						CASE_Z80_CB_OP1(0x00,Z80_RLC1); // RLC <o>
						CASE_Z80_CB_OP1(0x08,Z80_RRC1); // RRC <o>
						CASE_Z80_CB_OP1(0x10,Z80_RL1); // RL <o>
						CASE_Z80_CB_OP1(0x18,Z80_RR1); // RR <o>
						CASE_Z80_CB_OP1(0x20,Z80_SLA1); // SLA <o>
						CASE_Z80_CB_OP1(0x28,Z80_SRA1); // SRA <o>
						CASE_Z80_CB_OP1(0x30,Z80_SLL1); // SLL <o>
						CASE_Z80_CB_OP1(0x38,Z80_SRL1); // SRL <o>
						// 0xCB40-0xCB7F
						CASE_Z80_CB_BIT(0x40,0); // BIT 0,<o>
						CASE_Z80_CB_BIT(0x48,1); // BIT 1,<o>
						CASE_Z80_CB_BIT(0x50,2); // BIT 2,<o>
						CASE_Z80_CB_BIT(0x58,3); // BIT 3,<o>
						CASE_Z80_CB_BIT(0x60,4); // BIT 4,<o>
						CASE_Z80_CB_BIT(0x68,5); // BIT 5,<o>
						CASE_Z80_CB_BIT(0x70,6); // BIT 6,<o>
						CASE_Z80_CB_BIT(0x78,7); // BIT 7,<o>
						// 0xCB80-0xCBBF
						CASE_Z80_CB_OP2(0x80,Z80_RES1,0); // RES 0,<o>
						CASE_Z80_CB_OP2(0x88,Z80_RES1,1); // RES 1,<o>
						CASE_Z80_CB_OP2(0x90,Z80_RES1,2); // RES 2,<o>
						CASE_Z80_CB_OP2(0x98,Z80_RES1,3); // RES 3,<o>
						CASE_Z80_CB_OP2(0xA0,Z80_RES1,4); // RES 4,<o>
						CASE_Z80_CB_OP2(0xA8,Z80_RES1,5); // RES 5,<o>
						CASE_Z80_CB_OP2(0xB0,Z80_RES1,6); // RES 6,<o>
						CASE_Z80_CB_OP2(0xB8,Z80_RES1,7); // RES 7,<o>
						// 0xCBC0-0xCBFF
						CASE_Z80_CB_OP2(0xC0,Z80_SET1,0); // SET 0,<o>
						CASE_Z80_CB_OP2(0xC8,Z80_SET1,1); // SET 1,<o>
						CASE_Z80_CB_OP2(0xD0,Z80_SET1,2); // SET 2,<o>
						CASE_Z80_CB_OP2(0xD8,Z80_SET1,3); // SET 3,<o>
						CASE_Z80_CB_OP2(0xE0,Z80_SET1,4); // SET 4,<o>
						CASE_Z80_CB_OP2(0xE8,Z80_SET1,5); // SET 5,<o>
						CASE_Z80_CB_OP2(0xF0,Z80_SET1,6); // SET 6,<o>
						CASE_Z80_CB_OP2(0xF8,Z80_SET1,7); // SET 7,<o>
					}
					#undef CASE_Z80_CB_OP1
					#undef CASE_Z80_CB_BIT
					#undef CASE_Z80_CB_OP2
					break;
				case 0xDD: // PREFIX: XY SUBSET (IX)
				case 0xFD: // PREFIX: XY SUBSET (IY)
					{
						Z80_BEWARE; // see default case
						Z80W *xy=(op&0x20)?&z80_iy:&z80_ix;
						Z80_INC_R; op=Z80_FETCH;
						Z80_STRIDE(op+0x600); ++z80_pc.w;
						switch (op)
						{
							// 0xDD00-0xDD3F
							case 0x21: // LD IX,$NNNN
								Z80_LD2(xy->b);
								break;
							case 0x22: // LD ($NNNN),IX
								Z80_WR2(xy->b);
								break;
							case 0x2A: // LD IX,($NNNN)
								Z80_RD2(xy->b);
								break;
							case 0x03: // *INC BC
								++z80_bc.w; Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0x13: // *INC DE
								++z80_de.w; Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0x23: // INC IX
								++xy->w; Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0x33: // *INC SP
								++z80_sp.w; Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0x0B: // *DEC BC
								--z80_bc.w; Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0x1B: // *DEC DE
								--z80_de.w; Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0x2B: // DEC IX
								--xy->w; Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0x3B: // *DEC SP
								--z80_sp.w; Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0x24: // INC XH
								Z80_INC1(xy->b.h);
								break;
							case 0x25: // DEC XH
								Z80_DEC1(xy->b.h);
								break;
							case 0x2C: // INC XL
								Z80_INC1(xy->b.l);
								break;
							case 0x2D: // DEC XL
								Z80_DEC1(xy->b.l);
								break;
							case 0x34: // INC (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_INC1(b); Z80_IORQ_NEXT(1); Z80_WR_WZ; }
								break;
							case 0x35: // DEC (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_DEC1(b); Z80_IORQ_NEXT(1); Z80_WR_WZ; }
								break;
							case 0x26: // LD XH,$NN
								xy->b.h=Z80_RD_PC; ++z80_pc.w;
								break;
							case 0x2E: // LD XL,$NN
								xy->b.l=Z80_RD_PC; ++z80_pc.w;
								break;
							case 0x36: // LD (IX+$XX),$NN
								{ Z80_WZ_XY_1X(2); BYTE b=Z80_RD_PC; Z80_POKE(z80_wz,b); ++z80_pc.w; }
								break;
							case 0x09: // ADD IX,BC
								Z80_ADD2(xy->w,z80_bc.w);
								break;
							case 0x19: // ADD IX,DE
								Z80_ADD2(xy->w,z80_de.w);
								break;
							case 0x29: // ADD IX,IX
								Z80_ADD2(xy->w,xy->w);
								break;
							case 0x39: // ADD IX,SP
								Z80_ADD2(xy->w,z80_sp.w);
								break;
							// 0xDD40-0xDD7F
							case 0x44: // LD B,XH
								z80_bc.b.h=xy->b.h;
								break;
							case 0x45: // LD B,XL
								z80_bc.b.h=xy->b.l;
								break;
							case 0x46: // LD B,(IX+$XX)
								Z80_WZ_XY_1X(5); z80_bc.b.h=Z80_PEEK(z80_wz);
								break;
							case 0x4C: // LD C,XH
								z80_bc.b.l=xy->b.h;
								break;
							case 0x4D: // LD C,XL
								z80_bc.b.l=xy->b.l;
								break;
							case 0x4E: // LD C,(IX+$XX)
								Z80_WZ_XY_1X(5); z80_bc.b.l=Z80_PEEK(z80_wz);
								break;
							case 0x54: // LD D,XH
								z80_de.b.h=xy->b.h;
								break;
							case 0x55: // LD D,XL
								z80_de.b.h=xy->b.l;
								break;
							case 0x56: // LD D,(IX+$XX)
								Z80_WZ_XY_1X(5); z80_de.b.h=Z80_PEEK(z80_wz);
								break;
							case 0x5C: // LD E,XH
								z80_de.b.l=xy->b.h;
								break;
							case 0x5D: // LD E,XL
								z80_de.b.l=xy->b.l;
								break;
							case 0x5E: // LD E,(IX+$XX)
								Z80_WZ_XY_1X(5); z80_de.b.l=Z80_PEEK(z80_wz);
								break;
							case 0x60: // LD XH,B
								xy->b.h=z80_bc.b.h;
								break;
							case 0x61: // LD XH,C
								xy->b.h=z80_bc.b.l;
								break;
							case 0x62: // LD XH,D
								xy->b.h=z80_de.b.h;
								break;
							case 0x63: // LD XH,E
								xy->b.h=z80_de.b.l;
								break;
							case 0x65: // LD XH,XL
								xy->b.h=xy->b.l;
								break;
							case 0x66: // LD H,(IX+$XX)
								Z80_WZ_XY_1X(5); z80_hl.b.h=Z80_PEEK(z80_wz);
								break;
							case 0x67: // LD XH,A
								xy->b.h=z80_af.b.h;
								break;
							case 0x68: // LD XL,B
								xy->b.l=z80_bc.b.h;
								break;
							case 0x69: // LD XL,C
								xy->b.l=z80_bc.b.l;
								break;
							case 0x6A: // LD XL,D
								xy->b.l=z80_de.b.h;
								break;
							case 0x6B: // LD XL,E
								xy->b.l=z80_de.b.l;
								break;
							case 0x6C: // LD XL,XH
								xy->b.l=xy->b.h;
								break;
							case 0x6E: // LD L,(IX+$XX)
								Z80_WZ_XY_1X(5); z80_hl.b.l=Z80_PEEK(z80_wz);
								break;
							case 0x6F: // LD XL,A
								xy->b.l=z80_af.b.h;
								break;
							case 0x70: // LD (IX+$XX),B
								Z80_WZ_XY_1X(5); Z80_POKE(z80_wz,z80_bc.b.h);
								break;
							case 0x71: // LD (IX+$XX),C
								Z80_WZ_XY_1X(5); Z80_POKE(z80_wz,z80_bc.b.l);
								break;
							case 0x72: // LD (IX+$XX),D
								Z80_WZ_XY_1X(5); Z80_POKE(z80_wz,z80_de.b.h);
								break;
							case 0x73: // LD (IX+$XX),E
								Z80_WZ_XY_1X(5); Z80_POKE(z80_wz,z80_de.b.l);
								break;
							case 0x74: // LD (IX+$XX),H
								Z80_WZ_XY_1X(5); Z80_POKE(z80_wz,z80_hl.b.h);
								break;
							case 0x75: // LD (IX+$XX),L
								Z80_WZ_XY_1X(5); Z80_POKE(z80_wz,z80_hl.b.l);
								break;
							case 0x77: // LD (IX+$XX),A
								Z80_WZ_XY_1X(5); Z80_POKE(z80_wz,z80_af.b.h);
								break;
							case 0x7C: // LD A,XH
								z80_af.b.h=xy->b.h;
								break;
							case 0x7D: // LD A,XL
								z80_af.b.h=xy->b.l;
								break;
							case 0x7E: // LD A,(IX+$XX)
								Z80_WZ_XY_1X(5); z80_af.b.h=Z80_PEEK(z80_wz);
							case 0x64: // LD XH,XH
								//xy->b.h=xy->b.h;
								//break;
							case 0x6D: // LD XL,XL
								//xy->b.l=xy->b.l;
								//break;
								break;
							// 0xDD80-0xDDBF
							case 0x84: // ADD XH
								Z80_ADD1(xy->b.h);
								break;
							case 0x85: // ADD XL
								Z80_ADD1(xy->b.l);
								break;
							case 0x86: // ADD (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_ADD1(b); }
								break;
							case 0x8C: // ADC XH
								Z80_ADC1(xy->b.h);
								break;
							case 0x8D: // ADC XL
								Z80_ADC1(xy->b.l);
								break;
							case 0x8E: // ADC (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_ADC1(b); }
								break;
							case 0x94: // SUB XH
								Z80_SUB1(xy->b.h);
								break;
							case 0x95: // SUB XL
								Z80_SUB1(xy->b.l);
								break;
							case 0x96: // SUB (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_SUB1(b); }
								break;
							case 0x9C: // SBC XH
								Z80_SBC1(xy->b.h);
								break;
							case 0x9D: // SBC XL
								Z80_SBC1(xy->b.l);
								break;
							case 0x9E: // SBC (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_SBC1(b); }
								break;
							case 0xA4: // AND XH
								Z80_AND1(xy->b.h);
								break;
							case 0xA5: // AND XL
								Z80_AND1(xy->b.l);
								break;
							case 0xA6: // AND (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_AND1(b); }
								break;
							case 0xAC: // XOR XH
								Z80_XOR1(xy->b.h);
								break;
							case 0xAD: // XOR XL
								Z80_XOR1(xy->b.l);
								break;
							case 0xAE: // XOR (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_XOR1(b); }
								break;
							case 0xB4: // OR XH
								Z80_OR1(xy->b.h);
								break;
							case 0xB5: // OR XL
								Z80_OR1(xy->b.l);
								break;
							case 0xB6: // OR (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_OR1(b); }
								break;
							case 0xBC: // CP XH
								Z80_CP1(xy->b.h);
								break;
							case 0xBD: // CP XL
								Z80_CP1(xy->b.l);
								break;
							case 0xBE: // CP (IX+$XX)
								{ Z80_WZ_XY_1X(5); Z80_RD_WZ; Z80_CP1(b); }
								break;
							// 0xDDC0-0xDDFF
							case 0xC0: // *RET NZ
								Z80_WAIT(1); if (!(z80_af.b.l&0x40))
								{
									Z80_RET2;
									Z80_STRIDE(0x1C0);
								}
								else
									Z80_STRIDE_1;
								break;
							case 0xC8: // *RET Z
								Z80_WAIT(1); if (z80_af.b.l&0x40)
								{
									Z80_RET2;
									Z80_STRIDE(0x1C8);
								}
								else
									Z80_STRIDE_1;
								break;
							case 0xD0: // *RET NC
								Z80_WAIT(1); if (!(z80_af.b.l&0x01))
								{
									Z80_RET2;
									Z80_STRIDE(0x1D0);
								}
								else
									Z80_STRIDE_1;
								break;
							case 0xD8: // *RET C
								Z80_WAIT(1); if (z80_af.b.l&0x01)
								{
									Z80_RET2;
									Z80_STRIDE(0x1D8);
								}
								else
									Z80_STRIDE_1;
								break;
							case 0xE0: // *RET NV
								Z80_WAIT(1); if (!(z80_af.b.l&0x04))
								{
									Z80_RET2;
									Z80_STRIDE(0x1E0);
								}
								else
									Z80_STRIDE_1;
								break;
							case 0xE8: // *RET V
								Z80_WAIT(1); if (z80_af.b.l&0x04)
								{
									Z80_RET2;
									Z80_STRIDE(0x1E8);
								}
								else
									Z80_STRIDE_1;
								break;
							case 0xF0: // *RET NS
								Z80_WAIT(1); if (!(z80_af.b.l&0x80))
								{
									Z80_RET2;
									Z80_STRIDE(0x1F0);
								}
								else
									Z80_STRIDE_1;
								break;
							case 0xF8: // *RET S
								Z80_WAIT(1); if (z80_af.b.l&0x80)
								{
									Z80_RET2;
									Z80_STRIDE(0x1F8);
								}
								else
									Z80_STRIDE_1;
								break;
							case 0xE1: // POP IX
								Z80_POP2(xy->b);
								break;
							case 0xE5: // PUSH IX
								Z80_WAIT(1);
								Z80_PUSH2(xy->b);
								break;
							case 0xE3: // EX IX,(SP)
								z80_wz=Z80_PEEK(z80_sp.w);
								++z80_sp.w;
								z80_wz+=Z80_PEEKX(z80_sp.w)<<8;
								Z80_POKE1(z80_sp.w,xy->b.h);
								--z80_sp.w;
								Z80_POKE2(z80_sp.w,xy->b.l);
								xy->w=z80_wz;
								Z80_IORQ_1X_NEXT(2);
								Z80_STRIDE_1;
								break;
							case 0xE9: // JP IX
								z80_pc.w=xy->w;
								break;
							case 0xF9: // LD SP,IX
								z80_sp.w=xy->w;
								Z80_STRIDE_1;
								Z80_WAIT(2);
								break;
							case 0xCB: // PREFIX: XYCB SUBSET
								{
									Z80_WZ_XY;
									op=Z80_RD_PC;
									Z80_STRIDE(op+0x500);
									Z80_IORQ_1X_NEXT(2);
									Z80_RD_WZ; Z80_IORQ_NEXT(1);
									++z80_pc.w;
									// the XYCB set is extremely repetitive and thus worth abridging
									#define CASE_Z80_XYCB_OP1(xx,yy) \
										case xx+0: yy; Z80_WR_WZ; z80_bc.b.h=b; break; case xx+1: yy; Z80_WR_WZ; z80_bc.b.l=b; break; \
										case xx+2: yy; Z80_WR_WZ; z80_de.b.h=b; break; case xx+3: yy; Z80_WR_WZ; z80_de.b.l=b; break; \
										case xx+4: yy; Z80_WR_WZ; z80_hl.b.h=b; break; case xx+5: yy; Z80_WR_WZ; z80_hl.b.l=b; break; \
										case xx+7: yy; Z80_WR_WZ; z80_af.b.h=b; break; case xx+6: yy; Z80_WR_WZ; break
									#define CASE_Z80_XYCB_BIT(xx,yy) \
										case xx+6: Z80_BIT1(yy,b,z80_wz>>8); break; \
										case xx+0: case xx+1: case xx+2: case xx+3: case xx+4: case xx+5: case xx+7: Z80_BIT1(yy,b,b); break
									switch (op)
									{
										// 0xDDCBXX00-0xDDCBXX3F
										CASE_Z80_XYCB_OP1(0x00,Z80_RLC1(b)); // RLC (IX+$XX)
										CASE_Z80_XYCB_OP1(0x08,Z80_RRC1(b)); // RRC (IX+$XX)
										CASE_Z80_XYCB_OP1(0x10,Z80_RL1(b)); // RL (IX+$XX)
										CASE_Z80_XYCB_OP1(0x18,Z80_RR1(b)); // RR (IX+$XX)
										CASE_Z80_XYCB_OP1(0x20,Z80_SLA1(b)); // SLA (IX+$XX)
										CASE_Z80_XYCB_OP1(0x28,Z80_SRA1(b)); // SRA (IX+$XX)
										CASE_Z80_XYCB_OP1(0x30,Z80_SLL1(b)); // SLL (IX+$XX)
										CASE_Z80_XYCB_OP1(0x38,Z80_SRL1(b)); // SRL (IX+$XX)
										// 0xDDCBXX40-0xDDCBXX7F
										CASE_Z80_XYCB_BIT(0x40,0); // BIT 0,(IX+$XX)
										CASE_Z80_XYCB_BIT(0x48,1); // BIT 1,(IX+$XX)
										CASE_Z80_XYCB_BIT(0x50,2); // BIT 2,(IX+$XX)
										CASE_Z80_XYCB_BIT(0x58,3); // BIT 3,(IX+$XX)
										CASE_Z80_XYCB_BIT(0x60,4); // BIT 4,(IX+$XX)
										CASE_Z80_XYCB_BIT(0x68,5); // BIT 5,(IX+$XX)
										CASE_Z80_XYCB_BIT(0x70,6); // BIT 6,(IX+$XX)
										CASE_Z80_XYCB_BIT(0x78,7); // BIT 7,(IX+$XX)
										// 0xDDCBXX80-0xDDCBXXBF
										CASE_Z80_XYCB_OP1(0x80,Z80_RES1(0,b)); // RES 0,(IX+$XX)
										CASE_Z80_XYCB_OP1(0x88,Z80_RES1(1,b)); // RES 1,(IX+$XX)
										CASE_Z80_XYCB_OP1(0x90,Z80_RES1(2,b)); // RES 2,(IX+$XX)
										CASE_Z80_XYCB_OP1(0x98,Z80_RES1(3,b)); // RES 3,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xA0,Z80_RES1(4,b)); // RES 4,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xA8,Z80_RES1(5,b)); // RES 5,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xB0,Z80_RES1(6,b)); // RES 6,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xB8,Z80_RES1(7,b)); // RES 7,(IX+$XX)
										// 0xDDCBXXC0-0xDDCBXXFF
										CASE_Z80_XYCB_OP1(0xC0,Z80_SET1(0,b)); // SET 0,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xC8,Z80_SET1(1,b)); // SET 1,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xD0,Z80_SET1(2,b)); // SET 2,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xD8,Z80_SET1(3,b)); // SET 3,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xE0,Z80_SET1(4,b)); // SET 4,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xE8,Z80_SET1(5,b)); // SET 5,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xF0,Z80_SET1(6,b)); // SET 6,(IX+$XX)
										CASE_Z80_XYCB_OP1(0xF8,Z80_SET1(7,b)); // SET 7,(IX+$XX)
									}
									#undef CASE_Z80_XYCB_OP1
									#undef CASE_Z80_XYCB_BIT
									break;
								}
							default: // ILLEGAL DDXX/FDXX!
								z80_iff0=0; // delay interruption (if any)
								--z80_pc.w; Z80_DEC_R; // undo increases!!
								// Z80_STRIDE(op) MUST BE ZERO if OP is illegal!!
								Z80_REWIND; // terrible hack to undo the opcode fetching!
								break;
						}
					}
					break;
				case 0xED: // PREFIX: ED SUBSET
					Z80_INC_R; op=Z80_FETCH;
					Z80_STRIDE(op+0x200); ++z80_pc.w;
					switch (op)
					{
						// 0xED00-0xED3F
						// 0xED40-0xED7F
						case 0x40: // IN B,(C)
							Z80_IN2(z80_bc.b.h,0x340);
							break;
						case 0x48: // IN C,(C)
							Z80_IN2(z80_bc.b.l,0x348);
							break;
						case 0x50: // IN D,(C)
							Z80_IN2(z80_de.b.h,0x350);
							break;
						case 0x58: // IN E,(C)
							Z80_IN2(z80_de.b.l,0x358);
							break;
						case 0x60: // IN H,(C)
							Z80_IN2(z80_hl.b.h,0x360);
							break;
						case 0x68: // IN L,(C)
							Z80_IN2(z80_hl.b.l,0x368);
							break;
						case 0x70: // IN (C)
							{ BYTE dummy; Z80_IN2(dummy,0x370); }
							break;
						case 0x78: // IN A,(C)
							Z80_IN2(z80_af.b.h,0x378);
							break;
						case 0x41: // OUT (C),B
							Z80_OUT2(z80_bc.b.h,0x341);
							break;
						case 0x49: // OUT (C),C
							Z80_OUT2(z80_bc.b.l,0x349);
							break;
						case 0x51: // OUT (C),D
							Z80_OUT2(z80_de.b.h,0x351);
							break;
						case 0x59: // OUT (C),E
							Z80_OUT2(z80_de.b.l,0x359);
							break;
						case 0x61: // OUT (C),H
							Z80_OUT2(z80_hl.b.h,0x361);
							break;
						case 0x69: // OUT (C),L
							Z80_OUT2(z80_hl.b.l,0x369);
							break;
						case 0x71: // OUT (C)
							Z80_OUT2(z80_out0(),0x371);
							break;
						case 0x79: // OUT (C),A
							Z80_OUT2(z80_af.b.h,0x379);
							break;
						case 0x42: // SBC HL,BC
							Z80_SBC2(z80_bc);
							break;
						case 0x52: // SBC HL,DE
							Z80_SBC2(z80_de);
							break;
						case 0x62: // SBC HL,HL
							Z80_SBC2(z80_hl);
							break;
						case 0x72: // SBC HL,SP
							Z80_SBC2(z80_sp);
							break;
						case 0x4A: // ADC HL,BC
							Z80_ADC2(z80_bc);
							break;
						case 0x5A: // ADC HL,DE
							Z80_ADC2(z80_de);
							break;
						case 0x6A: // ADC HL,HL
							Z80_ADC2(z80_hl);
							break;
						case 0x7A: // ADC HL,SP
							Z80_ADC2(z80_sp);
							break;
						case 0x43: // LD ($NNNN),BC
							Z80_WR2(z80_bc.b);
							break;
						case 0x53: // LD ($NNNN),DE
							Z80_WR2(z80_de.b);
							break;
						case 0x63: // *LD ($NNNN),HL
							Z80_WR2(z80_hl.b);
							break;
						case 0x73: // LD ($NNNN),SP
							Z80_WR2(z80_sp.b);
							break;
						case 0x4B: // LD BC,($NNNN)
							Z80_RD2(z80_bc.b);
							break;
						case 0x5B: // LD DE,($NNNN)
							Z80_RD2(z80_de.b);
							break;
						case 0x6B: // *LD HL,($NNNN)
							Z80_RD2(z80_hl.b);
							break;
						case 0x7B: // LD SP,($NNNN)
							Z80_RD2(z80_sp.b);
							break;
						case 0x44: // NEG
						case 0x4C: // *NEG
						case 0x54: // *NEG
						case 0x5C: // *NEG
						case 0x64: // *NEG
						case 0x6C: // *NEG
						case 0x74: // *NEG
						case 0x7C: // *NEG
							{ BYTE b=z80_af.b.h; z80_af.b.h=0; Z80_SUB1(b); }
							break;
						case 0x45: // RETN
						case 0x55: // *RETN
						case 0x65: // *RETN
						case 0x75: // *RETN
							z80_iff.b.l=z80_iff.b.h;
							z80_retn();
							// no `break`!
						case 0x4D: // RETI
						case 0x5D: // *RETI
						case 0x6D: // *RETI
						case 0x7D: // *RETI
							Z80_RET2;
							break;
						case 0x46: // IM 0
						case 0x4E: // *IM 0
						case 0x66: // *IM 0
						case 0x6E: // *IM 0
							z80_imd=0;
							break;
						case 0x56: // IM 1
						case 0x76: // *IM 1
							z80_imd=1;
							break;
						case 0x5E: // IM 2
						case 0x7E: // *IM 2
							z80_imd=2;
							break;
						case 0x47: // LD I,A
							z80_ir.b.h=z80_af.b.h;
							Z80_STRIDE_1;
							Z80_WAIT(1);
							break;
						case 0x4F: // LD R,A
							r7=z80_ir.b.l=z80_af.b.h;
							Z80_STRIDE_1;
							Z80_WAIT(1);
							break;
						case 0x57: // LD A,I
							z80_af.b.l=(z80_af.b.l&1)+z80_flags_sgn[z80_af.b.h=z80_ir.b.h]+(z80_iff.b.h?0x04:0x00); // SZ000V0-
							Z80_STRIDE_1;
							Z80_WAIT(1);
							break;
						case 0x5F: // LD A,R
							z80_af.b.l=(z80_af.b.l&1)+z80_flags_sgn[z80_af.b.h=Z80_GET_R8]+(z80_iff.b.h?0x04:0x00); // SZ000V0-
							Z80_STRIDE_1;
							Z80_WAIT(1);
							break;
						case 0x77: // *LD I
						case 0x7F: // *LD R
							z80_af.b.l=(z80_af.b.l&0xFB)+(z80_iff.b.h?0x04:0x00); // -----V--
							break;
						case 0x67: // RRD
							{
								BYTE b=Z80_PEEK(z80_hl.w),z;
								Z80_IORQ_1X_NEXT(4);
								z=(b>>4)+(z80_af.b.h<<4); Z80_POKE(z80_hl.w,z);
								z80_af.b.h=(z80_af.b.h&0xF0)+(b&0x0F);
								z80_af.b.l=z80_flags_xor[z80_af.b.h]|(z80_af.b.l&0x01);
								z80_wz=z80_hl.w+1;
							}
							break;
						case 0x6F: // RLD
							{
								BYTE b=Z80_PEEK(z80_hl.w),z;
								Z80_IORQ_1X_NEXT(4);
								z=(b<<4)+(z80_af.b.h&0x0F); Z80_POKE(z80_hl.w,z);
								z80_af.b.h=(z80_af.b.h&0xF0)+(b>>4);
								z80_af.b.l=z80_flags_xor[z80_af.b.h]|(z80_af.b.l&0x01);
								z80_wz=z80_hl.w+1;
							}
							break;
						// 0xED80-0xEDBF
						case 0xA0: // LDI
							Z80_STRIDE_1;
							Z80_LDID2;
							Z80_IORQ_1X_NEXT(2);
							++z80_hl.w; ++z80_de.w;
							break;
						case 0xA8: // LDD
							Z80_STRIDE_1;
							Z80_LDID2;
							Z80_IORQ_1X_NEXT(2);
							--z80_hl.w; --z80_de.w;
							break;
						case 0xB0: // LDIR
							Z80_STRIDE_1;
							Z80_LDID2;
							Z80_IORQ_1X_NEXT(2);
							 ++z80_hl.w; ++z80_de.w;
							if (z80_af.b.l&0x04)
							{
								Z80_IORQ_1X_NEXT(5);
								z80_wz=--z80_pc.w; --z80_pc.w;
								Z80_STRIDE(0x3B0);
							}
							break;
						case 0xB8: // LDDR
							Z80_STRIDE_1;
							Z80_LDID2;
							Z80_IORQ_1X_NEXT(2);
							--z80_hl.w; --z80_de.w;
							if (z80_af.b.l&0x04)
							{
								Z80_IORQ_1X_NEXT(5);
								z80_wz=--z80_pc.w; --z80_pc.w;
								Z80_STRIDE(0x3B8);
							}
							break;
						case 0xA1: // CPI
							Z80_CPID2; ++z80_hl.w; ++z80_wz;
							Z80_IORQ_1X_NEXT(5);
							break;
						case 0xA9: // CPD
							Z80_CPID2; --z80_hl.w; --z80_wz;
							Z80_IORQ_1X_NEXT(5);
							break;
						case 0xB1: // CPIR
							Z80_CPID2; ++z80_hl.w; ++z80_wz;
							Z80_IORQ_1X_NEXT(5);
							if ((z80_af.b.l&0x44)==0x04)
							{
								Z80_IORQ_1X_NEXT(5);
								z80_wz=--z80_pc.w; --z80_pc.w;
								Z80_STRIDE(0x3B1);
								Z80_STRIDE_1;
							}
							break;
						case 0xB9: // CPDR
							Z80_CPID2; --z80_hl.w; --z80_wz;
							Z80_IORQ_1X_NEXT(5);
							if ((z80_af.b.l&0x44)==0x04)
							{
								Z80_IORQ_1X_NEXT(5);
								z80_wz=--z80_pc.w; --z80_pc.w;
								Z80_STRIDE(0x3B9);
								Z80_STRIDE_1;
							}
							break;
						case 0xA2: // INI
							Z80_INID2(+1,0x3A2); ++z80_wz; ++z80_hl.w;
							break;
						case 0xAA: // IND
							Z80_INID2(-1,0x3AA); --z80_wz; --z80_hl.w;
							break;
						case 0xB2: // INIR
							Z80_INID2(+1,0x3A2); ++z80_wz; ++z80_hl.w;
							if (!(z80_af.b.l&0x40))
							{
								Z80_IORQ_1X_NEXT(5);
								z80_pc.w-=2;
								Z80_STRIDE(0x3B2);
							}
							break;
						case 0xBA: // INDR
							Z80_INID2(-1,0x3AA); --z80_wz; --z80_hl.w;
							if (!(z80_af.b.l&0x40))
							{
								Z80_IORQ_1X_NEXT(5);
								z80_pc.w-=2;
								Z80_STRIDE(0x3BA);
							}
							break;
						case 0xA3: // OUTI
							Z80_OTID2(+1,0x3A3); ++z80_wz; ++z80_hl.w;
							break;
						case 0xAB: // OUTD
							Z80_OTID2(-1,0x3AB); --z80_wz; --z80_hl.w;
							break;
						case 0xB3: // OTIR
							Z80_OTID2(+1,0x3A3); ++z80_wz; ++z80_hl.w;
							if (!(z80_af.b.l&0x40))
							{
								Z80_MREQ_1X(5,z80_hl.w);
								z80_pc.w-=2;
								Z80_STRIDE(0x3B3);
							}
							break;
						case 0xBB: // OTDR
							Z80_OTID2(-1,0x3AB); --z80_wz; --z80_hl.w;
							if (!(z80_af.b.l&0x40))
							{
								Z80_MREQ_1X(5,z80_hl.w);
								z80_pc.w-=2;
								Z80_STRIDE(0x3BB);
							}
							break;
						// 0xEDC0-0xEDFF
						default: // ILLEGAL EDXX!
							break;
					}
					break;
			}
		}
		#ifdef DEBUG
		if ((z80_pc.w==breakpoint)&&(breakpoint!=0xFFFF))
			session_signal|=SESSION_SIGNAL_DEBUG,breakpoint=0xFFFF;
		#else
		if (z80_breakpoints[z80_pc.w])
		{
			if (z80_breakpoints[z80_pc.w]&8) // log bytes?
			{
				if (!z80_debug_logfile) // try creating a log file?
				{
					char *s;
					if (s=session_newfile("","*","Register log file"))
						z80_debug_logfile=fopen(s,"wb"),z80_debug_logpos=0;
				}
				if (!z80_debug_logfile)
					z80_breakpoints[z80_pc.w]=0; // remove breakpoint on failure!
				else
				{
					int z;
					switch(z80_breakpoints[z80_pc.w]&7)
					{
						// Z80 8-bit "BCDEHLFA" order
						case 0: z=z80_bc.b.h; break;
						case 1: z=z80_bc.b.l; break;
						case 2: z=z80_de.b.h; break;
						case 3: z=z80_de.b.l; break;
						case 4: z=z80_hl.b.h; break;
						case 5: z=z80_hl.b.l; break;
						case 6: z=z80_af.b.l; break;
						case 7: z=z80_af.b.h; break;
					}
					z80_debug_logtmp[z80_debug_logpos]=z;
					if (++z80_debug_logpos>=length(z80_debug_logtmp))
						fwrite(z80_debug_logtmp,1,sizeof(z80_debug_logtmp),z80_debug_logfile),z80_debug_logpos=0;
				}
			}
			else
			{
				z80_debug_reset(); session_signal|=SESSION_SIGNAL_DEBUG; _t_=0; // break!
			}
		}
		#endif
	}
	while (z80_t<_t_);
	z80_ir.b.l=Z80_GET_R8; // unify R7+R8!
	z80_sync(z80_t); // flush accumulated T!
}

#ifdef DEBUG
	#define z80_debug_configread(i) (z80_debug_peekpoke=(i)) // store config
	#define z80_debug_configwrite() (z80_debug_peekpoke) // restore config
#else
char z80_debug_panel=0; // current panel: 0 disassembly, 1 registers, 2 memory, 3 stack
char z80_debug_page=0; // hardware info
char z80_debug_pnl0_x=0; // X position (word nibble)
char z80_debug_pnl1_x=0,z80_debug_pnl1_y=0; // X+Y position (nibble+register)
char z80_debug_pnl2_x=0; // X position (byte nibble)
WORD z80_debug_pnl2_w=0; // dump byte
char z80_debug_pnl3_x=0; // X position (word nibble)
WORD z80_debug_cache[16]; // used when scrolling down
int z80_debug_peek(int q,WORD m)
{
	return q?POKE(m):PEEK(m);
}
void z80_debug_show(void) // redraw debug screen
{
	int y,x;
	WORD m,w;
	memset(debug_buffer,' ',sizeof(debug_buffer)); // clear buffer
	z80_break_stack=0xFFFF; z80_breakpoints[z80_debug_volatile]&=~128; // reset volatile breakpoints
	// DISASSEMBLY
	m=z80_debug_pnl0_w;
	for (y=0;y<16;++y)
	{
		z80_debug_cache[y]=m;
		debug_locate(0,y);
		char *z=debug_output;
		x=z80_breakpoints[m];
		w=m-z80_pc.w;
		m=z80_dasm(session_tmpstr,m);
		debug_prints(session_tmpstr);
		if (!w)
			z[4]='#'; // current PC
		if (x)
		{
			if (x&8)
				z[5]="BCDEHLFA"[x&7]; // log
			else
				z[5]='@'; // breakpoint
		}
	}
	if (z80_debug_panel==0)
	{
		debug_locate(6+z80_debug_pnl0_x,0);
		*debug_output^=128;
	}
	// REGISTERS
	sprintf(session_tmpstr,"SZ5H:3VNC\n"
		"AF = %04X\nBC = %04X\nDE = %04X\nHL = %04X\n"
		"AF'= %04X\nBC'= %04X\nDE'= %04X\nHL'= %04X\n"
		"IX = %04X\nIY = %04X\nIR = %04X\nWZ = %04X\n"
		"PC = %04X\nSP = %04X\n"
		"IM %c %cI %c\n"
		,z80_af.w,z80_bc.w,z80_de.w,z80_hl.w
		,z80_af2.w,z80_bc2.w,z80_de2.w,z80_hl2.w
		,z80_ix.w,z80_iy.w,z80_ir.w,z80_wz
		,z80_pc.w,z80_sp.w
		,'0'+z80_imd,'D'+z80_iff.b.l,z80_irq?'*':'-');
	for (y=0;y<8;++y)
		if (!(z80_af.b.l&(128>>y)))
			session_tmpstr[y+(y>=4)]='-';
	debug_dumpxy(-9,0,session_tmpstr);
	if (z80_debug_panel==1)
	{
		debug_locate(z80_debug_pnl1_x-4,z80_debug_pnl1_y+1);
		*debug_output^=128;
	}
	// MEMORY DUMP
	m=(z80_debug_pnl2_w-((DEBUG_LENGTH_Y-17)/2)*16)&~15;
	for (y=17;y<DEBUG_LENGTH_Y;++y)
	{
		debug_locate(0,y);
		debug_printi("%04X:",m);
		for (x=0;x<16;++x)
		{
			BYTE b=z80_debug_peek(z80_debug_peekpoke,m);
			debug_locate(5+x*2,y);
			debug_printi("%02X",b);
			if (z80_debug_panel==2&&m==z80_debug_pnl2_w)
				debug_output[z80_debug_pnl2_x?-1:-2]^=128;
			debug_output[32-1-x]=(b>=32&&b<127)?b:'.';
			++m;
		}
		*debug_output=z80_debug_peekpoke?'\\':'/';
	}
	// STACK
	m=z80_debug_pnl3_w;//z80_sp.w;
	for (y=17;y<DEBUG_LENGTH_Y;++y)
	{
		debug_locate(-9,y);
		debug_printi("%04X",m);
		*debug_output++=(m==z80_sp.w)?'#':':';
		w=PEEK(m); ++m;
		w+=PEEK(m)<<8; ++m;
		debug_printi("%04X",w);
	}
	if (z80_debug_panel==3)
	{
		debug_locate(z80_debug_pnl3_x-4,17);
		*debug_output^=128;
	}
	//for (y=0;y<DEBUG_LENGTH_X;++y) debug_locate(y,16),*debug_output=160;//'-';
	//for (y=0;y<DEBUG_LENGTH_Y;++y) debug_locate(-10,y),*debug_output=160;//'|';
	//debug_locate(-10,16),*debug_output='+';
	z80_debug_hard(z80_debug_page,-9-2-20,0);
	debug_locate(-9-2-18,16-1); debug_prints("press H for help");
	onscreen_debug();
}

int z80_debug_hex(char c) // 0..15 OK, <0 ERROR!
{
	if (c>='0'&&c<='9')
		return c-'0';
	if (c>='A'&&c<='F')
		return c-'A'+10;
	if (c>='a'&&c<='f')
		return c-'a'+10;
	return -1;
}
int z80_debug_hexs(char *s)
{
	char c;
	int i=0;
	while ((c=z80_debug_hex(*s))>=0)
		i=(i<<4)+c,++s;
	return i;
}
int z80_debug_goto(int i) // <0 ERROR, >=0 OK
{
	sprintf(session_parmtr,"%04X",i);
	if (session_input(session_parmtr,"Go to")>0)
		return z80_debug_hexs(session_parmtr);
	return -1;
}
BYTE z80_debug_findhexa,z80_debug_search[STRMAX];
void z80_debug_find(int q)
{
	if (!z80_debug_search[0])
		return;
	WORD m=!z80_debug_panel?z80_debug_pnl0_w:z80_debug_pnl2_w,n;
	BYTE l,*s;
	while (++m)
	{
		s=z80_debug_search;
		l=*s++;
		n=m;
		if (z80_debug_findhexa)
			while (l)
				if (z80_debug_peek(q,n)!=*s)
					break;
				else
					++s,--l,++n;
		else
			while (l)
				if (ucase(z80_debug_peek(q,n))!=*s)
					break;
				else
					++s,--l,++n;
		if (!l)
			break;
	}
	if (m)
		switch(z80_debug_panel)
		{
			case 0: z80_debug_pnl0_x=0, z80_debug_pnl0_w=m; break;
			case 2: z80_debug_pnl2_x=0, z80_debug_pnl2_w=m; break;
		}
}

int z80_debug_user(int k) // returns 0 if NOTHING, !0 if SOMETHING
{
	int i;
	if (k==13) // RETURN: STEP OVER
	{
		i=0;
		switch (PEEK(z80_pc.w))
		{
			case 0xCD: // CALL nnnn
			case 0xC4: case 0xCC: // CALL CC,nnnn
			case 0xD4: case 0xDC:
			case 0xE4: case 0xEC:
			case 0xF4: case 0xFC:
				i=3;
				break;
			case 0x76: // HALT
			case 0xC7: case 0xCF: // RST n
			case 0xD7: case 0xDF:
			case 0xE7: case 0xEF:
			case 0xF7: case 0xFF:
				i=1;
				break;
			case 0x10: // DJNZ rr
				i=2;
				break;
			case 0xED:
				if ((k=PEEK((WORD)(z80_pc.w+1)))>=0xB0&&k<0xC0) // LDxR / CPxR / INxR / OTxR
					i=2;
				break;
		}
		if (i)
		{
			z80_breakpoints[z80_debug_volatile=(WORD)(z80_pc.w+i)]|=128; // set volatile breakpoint
			session_signal&=~SESSION_SIGNAL_DEBUG;
			k=0;
		}
		else
			k=' '; // equivalent to SPACE
	}
	switch(k)
	{
		case 'U': // 'U': RETURN FROM..
			if (z80_sp.w>=0xFFFE) // fringe case?
				z80_breakpoints[z80_debug_volatile=PEEKW(z80_sp.w)]|=128; // set volatile breakpoint
			else
				z80_break_stack=z80_sp.w; // no `break`!
		case 27: // ESCAPE
			session_signal&=~SESSION_SIGNAL_DEBUG;
		case 0: // dummy!
			break;
		#if Z80_DEBUG_MMU
		case 'M': // MEMORY
			z80_debug_peekpoke=!z80_debug_peekpoke; break;
		#endif
		case 'V': // VIDEO
			++onscreen_debug_mask; break;
		case ' ': // SPACE: STEP INTO
			z80_main(1),z80_debug_reset(); break;
		case 160: // SHIFT+SPACE: STEP INTO (scanline)
			session_signal&=~SESSION_SIGNAL_DEBUG,z80_main(Z80_DEBUG_SCAN),
			session_signal|=SESSION_SIGNAL_DEBUG;z80_debug_reset(); break;
		case 7: // SHIFT+TAB
			z80_debug_panel=(z80_debug_panel-1)&3; break;
		case 12: // TAB
			z80_debug_panel=(z80_debug_panel+1)&3; break;
		case 'S': // SEARCH
			if (z80_debug_panel==0||z80_debug_panel==2)
			{
				strcpy(session_parmtr,debug_search);
				if (session_input(session_parmtr,"Search")>0)
				{
					strcpy(debug_search,session_parmtr);
					int i=0,l=0,c=1;
					char *s=session_parmtr,*t=z80_debug_search;
					if (z80_debug_findhexa=(*s=='$')) // hexadecimal
						while ((i=z80_debug_hex(*++s))>=0)
							((c=c*16+i)>=256)&&(*++t=c,++l,c=1);
					else // ASCII
						while (*++t=ucase(*s++))
							++l;
					z80_debug_search[0]=l;
					z80_debug_find(z80_debug_panel==2&&z80_debug_peekpoke);
				}
			}
			break;
		case 'N': // NEXT SEARCH
			if (z80_debug_panel==0||z80_debug_panel==2)
				z80_debug_find(z80_debug_panel==2&&z80_debug_peekpoke);
			break;
		case 'I': // INPUT FILE
			if ((i=z80_debug_pnl0_w,z80_debug_panel==0)||(i=z80_debug_pnl2_w,z80_debug_panel==2))
			{
				WORD w=i;
				char *s; FILE *f;
				if (s=session_getfile("","*","Input file"))
					if (f=fopen(s,"rb"))
					{
						while ((i=fgetc(f))>=0)
							POKE(w)=i,++w;
						fclose(f);
					}
			}
			break;
		case 'O': // OUTPUT FILE
			if ((i=z80_debug_pnl0_w,z80_debug_panel==0)||(i=z80_debug_pnl2_w,z80_debug_panel==2))
			{
				WORD w=i;
				char *s; FILE *f;
				session_parmtr[0]=0;
				if (session_input(session_parmtr,"Output length")>=0)
					if ((i=z80_debug_hexs(session_parmtr))>0)
						if (s=session_newfile("","*","Output file"))
							if (f=fopen(s,"wb"))
							{
								while (i--)
									fputc(PEEK(w),f),++w;
								fclose(f);
							}
			}
			break;
		case 'P': // PRINT DISASSEMBLY
			if (z80_debug_panel==0)
			{
				WORD w=z80_debug_pnl0_w;
				char *s; FILE *f;
				session_parmtr[0]=0;
				if (session_input(session_parmtr,"Disassembly length")>=0)
					if ((i=z80_debug_hexs(session_parmtr))>0)
						if (s=session_newfile("","*.TXT","Print disassembly"))
							if (f=fopen(s,"w"))
							{
								i+=w;
								while (w<i||w>=i+4) // WRAP!
									w=z80_dasm(session_tmpstr,w),fprintf(f,session_tmpstr),fputc('\n',f);
								fclose(f);
							}
			}
			break;
		case 'X': // EXTENDED HARDWARE INFO
			++z80_debug_page;
			break;
		case 'H': // HELP
			session_message(
				"Cursors\tNavigate panel\n"
				"Tab\tNext panel (shift: previous)\n"
				"0-9,A-F\tEdit hexadecimal value\n"
				"G\tGo to ADDRESS\n"
				"H\tHelp..\n"
				"I\tInput all bytes from FILE\n"
				"J\tJump to..\n"
				"K\tClose log file\n"
				"L\tLog 8-bit REGISTER into FILE\n"
				#if Z80_DEBUG_MMU
				"M\tToggle memory dump R/W map\n"
				#else
				//"M\t-\n"
				#endif
				"N\tNext search\n"
				"O\tOutput LENGTH bytes into FILE\n"
				"P\tPrint disassembly of LENGTH bytes into FILE\n"
				//"Q\t-\n"
				"R\tRun to..\n"
				"S\tSearch for STRING ($+string: hexadecimal)\n"
				"T\tToggle breakpoint\n"
				"U\tReturn from..\n"
				"V\tToggle debugger appearance\n"
				//"W\t-\n"
				#if Z80_DEBUG_EXT
				"X\tShow more hardware info\n"
				#else
				//"X\t-\n"
				#endif
				//"Y\t-\n"
				"Z\tDelete all breakpoints\n"
				"Space\tStep into..\n"
				"Return\tStep over..\n"
				"Escape\tExit\n"
				,"Debugger help"
				);
			break;
		default: switch(z80_debug_panel)
		{
			case 0: // DISASSEMBLY
				if ((i=z80_debug_hex(k))>=0)
				{
					WORD o=(z80_debug_pnl0_w+z80_debug_pnl0_x/2);
					i=(z80_debug_pnl0_x&1)?((PEEK(o)&240)+i):(i*16+(PEEK(o)&15));
					POKE(o)=i;
					k=128+9; // force RIGHT + RECALCULATE
				}
				i=0;
				switch(k)
				{
					case 8: if (--z80_debug_pnl0_x<0) z80_debug_pnl0_x&=1,--z80_debug_pnl0_w; break;
					case 128+9: z80_debug_cache[1]=z80_dasm(session_tmpstr,z80_debug_pnl0_w); // no 'break'!
					case 9: if ((WORD)(++z80_debug_pnl0_x/2+z80_debug_pnl0_w)==z80_debug_cache[1])
							z80_debug_pnl0_x&=1,z80_debug_pnl0_w=z80_debug_cache[1]; break;
					case 11: i=-1; break;
					case 10: i=1; break;
					case 28: z80_debug_pnl0_w=0,z80_debug_pnl0_x=0; break;
					case 29: z80_debug_pnl0_w=z80_pc.w,z80_debug_pnl0_x=0; break;
					case 31: i=1-16; break;
					case 30: i=16-1; break;
					case 'G': if ((i=z80_debug_goto(z80_debug_pnl0_w))>=0)
							z80_debug_pnl0_x=0,z80_debug_pnl0_w=i;
						break;
					case 'T': // TOGGLE BREAKPOINT
						z80_breakpoints[z80_debug_pnl0_w]=!z80_breakpoints[z80_debug_pnl0_w];
						break;
					case 'Z': // RESET BREAKPOINTS
						MEMZERO(z80_breakpoints);
						break;
					case 'J': // JUMP TO
						z80_pc.w=z80_debug_pnl0_w;
						break;
					case 'R': // RUN TO
						z80_breakpoints[z80_debug_volatile=z80_debug_pnl0_w]|=128; // set volatile breakpoint
						session_signal&=~SESSION_SIGNAL_DEBUG;
						break;
					case 'L': // LOG REGISTER
						session_parmtr[0]=0;
						if (session_input(session_parmtr,"Log register")==1)
						{
							switch(session_parmtr[0]&~32)
							{
								// cfr. "BCDEHLFA"
								case 'B': i=8+0; break;
								case 'C': i=8+1; break;
								case 'D': i=8+2; break;
								case 'E': i=8+3; break;
								case 'H': i=8+4; break;
								case 'L': i=8+5; break;
								case 'F': i=8+6; break;
								case 'A': i=8+7; break;
								default : i=0+0; break;
							}
							if (i)
								z80_breakpoints[z80_debug_pnl0_w]=i;
						}
						break;
					case 'K': // CLOSE LOG
						z80_debug_close(); // *!*
						break;
					default: k=0; break;
				}
				if (i&&k>0&&k<32) // cursors may need to move up or down
				{
					if (i>0) // DOWN
						z80_debug_pnl0_w=z80_debug_cache[i];
					else while (i++) // UP
					{
						WORD o=z80_debug_pnl0_w-4,p=z80_debug_pnl0_w-1; // longest+shortest possible operation
						while (o!=p&&z80_dasm(session_tmpstr,o)!=z80_debug_pnl0_w)
							++o;
						z80_debug_pnl0_w=o;
					}
					z80_dasm(session_tmpstr,z80_debug_pnl0_w);
					while (session_tmpstr[6+z80_debug_pnl0_x]==' ') // *!* dirty way to ensure that the cursor is valid!
						z80_debug_pnl0_x-=2;
				}
				break;
			case 1: // REGISTERS
				if ((i=z80_debug_hex(k))>=0)
				{
					WORD *o;
					switch (z80_debug_pnl1_y)
					{
						case  0: o=&z80_af .w; break;
						case  1: o=&z80_bc .w; break;
						case  2: o=&z80_de .w; break;
						case  3: o=&z80_hl .w; break;
						case  4: o=&z80_af2.w; break;
						case  5: o=&z80_bc2.w; break;
						case  6: o=&z80_de2.w; break;
						case  7: o=&z80_hl2.w; break;
						case  8: o=&z80_ix .w; break;
						case  9: o=&z80_iy .w; break;
						case 10: o=&z80_ir .w; break;
						case 11: o=&z80_wz   ; break;
						case 12: o=&z80_pc .w; break;
						default: o=&z80_sp .w; break;
					}
					switch (z80_debug_pnl1_x)
					{
						case 0 : *o=(*o&0x0FFF)+(i<<12); break;
						case 1 : *o=(*o&0xF0FF)+(i<< 8); break;
						case 2 : *o=(*o&0xFF0F)+(i<< 4); break;
						default: *o=(*o&0xFFF0)+(i<< 0); break;
					}
					k=9; // force RIGHT
				}
				switch(k)
				{
					case 8: if (--z80_debug_pnl1_x<0) z80_debug_pnl1_x=3,--z80_debug_pnl1_y; break; // wrap left and up
					case 9: if (++z80_debug_pnl1_x>3) z80_debug_pnl1_x=0,++z80_debug_pnl1_y; break; // wrap right and down
					case 11: --z80_debug_pnl1_y; break;
					case 10: ++z80_debug_pnl1_y; break;
					case 28: z80_debug_pnl1_x=0; break;
					case 29: z80_debug_pnl1_x=3; break;
					case 31: z80_debug_pnl1_y= 0; break;
					case 30: z80_debug_pnl1_y=13; break;
					default: k=0; break;
				}
				if (z80_debug_pnl1_y>13) z80_debug_pnl1_y=0; // wrap up
				else if (z80_debug_pnl1_y<0) z80_debug_pnl1_y=13; // wrap down
				break;
			case 2: // MEMORY DUMP
				if ((i=z80_debug_hex(k))>=0)
				{
					k=z80_debug_peek(z80_debug_peekpoke,z80_debug_pnl2_w);
					i=z80_debug_pnl2_x?((k&240)+i):(i*16+(k&15));
					POKE(z80_debug_pnl2_w)=i;
					k=9; // force RIGHT
				}
				switch(k)
				{
					case 8: if (--z80_debug_pnl2_x<0) z80_debug_pnl2_x=1,--z80_debug_pnl2_w; break; // prev byte
					case 9: if (++z80_debug_pnl2_x>1) z80_debug_pnl2_x=0,++z80_debug_pnl2_w; break; // next byte
					case 10: z80_debug_pnl2_w+=16; break;
					case 11: z80_debug_pnl2_w-=16; break;
					case 28: z80_debug_pnl2_x=0,z80_debug_pnl2_w=0; break;
					case 29: z80_debug_pnl2_x=0,z80_debug_pnl2_w=z80_pc.w; break;
					case 31: z80_debug_pnl2_w-=(DEBUG_LENGTH_Y-16-2)*16; break;
					case 30: z80_debug_pnl2_w+=(DEBUG_LENGTH_Y-16-2)*16; break;
					case 'G': if ((i=z80_debug_goto(z80_debug_pnl2_w))>=0) z80_debug_pnl2_x=0,z80_debug_pnl2_w=i; break;
					default: k=0; break;
				}
				break;
			case 3: // STACK
				if ((i=z80_debug_hex(k))>=0)
				{
					WORD o=PEEKW(z80_debug_pnl3_w);
					switch (z80_debug_pnl3_x)
					{
						case 0 : o=(o&0x0FFF)+(i<<12); break;
						case 1 : o=(o&0xF0FF)+(i<< 8); break;
						case 2 : o=(o&0xFF0F)+(i<< 4); break;
						default: o=(o&0xFFF0)+(i<< 0); break;
					}
					POKEW(z80_debug_pnl3_w,o);
					k=9; // force RIGHT
				}
				switch(k)
				{
					case 8: if (--z80_debug_pnl3_x<0) z80_debug_pnl3_x=3,z80_debug_pnl3_w-=2; break; // prev word
					case 9: if (++z80_debug_pnl3_x>3) z80_debug_pnl3_x=0,z80_debug_pnl3_w+=2; break; // next word
					case 10: z80_debug_pnl3_w+=2; break;
					case 11: z80_debug_pnl3_w-=2; break;
					case 28: z80_debug_pnl3_x=0,z80_debug_pnl3_w=0; break;
					case 29: z80_debug_pnl3_x=0,z80_debug_pnl3_w=z80_sp.w; break;
					case 31: z80_debug_pnl3_w-=2*(DEBUG_LENGTH_Y-16-2); break;
					case 30: z80_debug_pnl3_w+=2*(DEBUG_LENGTH_Y-16-2); break;
					case 'G': if ((i=z80_debug_goto(z80_debug_pnl3_w))>=0) z80_debug_pnl3_x=0,z80_debug_pnl3_w=i; break;
					default: k=0; break;
				}
				break;
		}
	}
	return k&&(debug_buffer[0]=128); // 128: draw debug screen
}
	#define z80_debug_configread(i) (onscreen_debug_mask=(i)/2,z80_debug_peekpoke=Z80_DEBUG_MMU?(i)&1:0)
	#define z80_debug_configwrite() onscreen_debug_mask*2+(z80_debug_peekpoke)
#endif

// ============================================= END OF Z80 EMULATION //
